/*
	WeaponSoul Tilemaps
	written by WolfCoder (2009)

	A tilemap is an array of square images typically used for 2d game maps
	Tilemaps z-coordinates are ignored as their collision hulls extend to infinity in the z direction
	Tilemaps are also entities, since they also have their own collision hulls
	Tilemaps may not have payloads, but they can have skills
*/

/* Includes */
#include "weaponsoul.h"

/* Macros */
#define COLLISION_BASE(var) sizeof(tilemap_header)+(var)->width*(var)->height*sizeof(unsigned int)

/* Globals */
unsigned long num_tilemaps;
int draw_hull = 0;
int tiles_begin;

/* Initializes the tilemaps */
void wstile_init()
{
	tiles_begin = sizeof(tilemap_header)/sizeof(unsigned int);
}

/* Handles the tilemap */
action wstile_action()
{
	/* Can't die inside the action */
	return 1;
}

/* Creates a new tilemap from the tileset given */
void wstile_create(image tileset,int width,int height)
{
	entity who;
	tilemap_header *data;
	/* Create the entity */
	wsaction_create(wstile_action);
	who = last;
	who->type = ENTITY_TILEMAP;
	/* Attach payload */
	wsaction_payload(who,wsimage_width(tileset)*wsimage_height(tileset)+width*height*sizeof(unsigned int)+sizeof(tilemap_header));
	/* Configure */
	data = who->payload;
	data->width = width;
	data->height = height;
	data->set = tileset;
}

/* Sets hull display */
void wstile_show_hull(int val)
{
	draw_hull = val;
}

/* Draws the tileset */
void wstile_draw(tilemap who)
{
	int bx,by;
	tilemap_header *data;
	unsigned int *tiles;
	unsigned char *colls;
	int tsx,tsy;
	int ox,oy;
	int fromx,fromy,tox,toy;
	/* Freeze position */
	ox = (int)(who->x-wscamera_draw_x());
	oy = (int)(who->y-wscamera_draw_y());
	/* Select */
	data = who->payload;
	tiles = (unsigned int*)who->payload;
	tiles +=tiles_begin;
	colls = (unsigned char *)tiles;
	colls += COLLISION_BASE(data);
	/* Measure */
	tsx = wsimage_width(data->set);
	tsy = wsimage_height(data->set);
	/* Find */
	fromx = (-ox)/tsx;
	if(fromx < 0)
		fromx = 0;
	fromy = (-oy)/tsy;
	if(fromy < 0)
		fromy = 0;
	tox = fromx+wsvideo_width()/tsx+1;
	if(tox > data->width)
		tox = data->width;
	toy = fromy+wsvideo_height()/tsy;
	if(toy > data->height)
		toy = data->height;
	/* Draw */
	for(by = fromy;by < toy;by++)
	{
		for(bx = fromx;bx < tox;bx++)
		{
			/* Read the tile */
			wsimage_frame(tiles[by*data->width+bx]);
			/* Highlight hulls */
			if(draw_hull)
			{
				if(colls[tiles[by*data->width+bx]])
					wsimage_color(1,0,0);
			}
			/* Then draw */
			wsimage_draw(data->set,bx*tsx+ox,by*tsy+oy);
		}
	}
}

/* Sets a tile at x and y IN the map to the tile specified */
void wstile_set(tilemap who,int x,int y,unsigned char tile)
{
	unsigned int *tiles;
	tilemap_header *data;
	/* Select */
	data = who->payload;
	/* Protect bounds */
	if(x < 0 || y < 0)
		return;
	if(x >= data->width)
		return;
	if(y >= data->height)
		return;
	/* Get tile base */
	tiles = (unsigned int*)who->payload;
	tiles += tiles_begin;
	/* Set */
	tiles[y*data->width+x] = tile;
}

/* Sets a collision value for a specific tile */
void wstile_collision_set(tilemap who,int tile,unsigned char value)
{
	unsigned char *colls;
	tilemap_header *data;
	/* Select */
	data = who->payload;
	/* Get tile base */
	colls = (unsigned char*)who->payload;
	colls += COLLISION_BASE(data);
	/* Set */
	colls[tile] = value;
}

/* Returns the collision code for the tilemap at tile x and y */
int wstile_collision_get(tilemap who,int x,int y)
{
	unsigned int *tiles;
	unsigned char *colls;
	tilemap_header *data;
	/* Select */
	data = who->payload;
	/* Out of bounds */
	if(x < 0 || x >= data->width)
		return 0;
	if(y < 0 || y >= data->height)
		return 0;
	/* Select tile and coll data */
	tiles = (unsigned int*)who->payload;
	colls = (unsigned char*)who->payload;
	tiles += tiles_begin;
	colls += COLLISION_BASE(data);
	/* Returnt the collision status of the tile at x and y */
	return colls[tiles[y*data->width+x]];
}
	
