/* Mode System */

/* Globals */
void (*mode_init[MAX_MODES])();
void (*mode_main[MAX_MODES])();
void (*mode_exit[MAX_MODES])();
void (*mode_2d[MAX_MODES])();
int mode_global[MAX_MODES];
unsigned long num_modes = 0;
unsigned long current_mode;
unsigned long changing_modes;
unsigned long target_mode;
unsigned long num_2d_modes;
int global_ready = 0;

/* Initialize the mode system */
void wsmode_init()
{
	/* Initialize variables */
	current_mode = 0;
	changing_modes = 1;
	target_mode = 0;
	/* Announce */
	wslog_string("[MODE] ");
	wslog_num(num_modes);
	wslog_string(" modes created.\n");
	wslog_string("\t");
	wslog_num(num_2d_modes);
	wslog_string(" are 2D modes.\n");
}

/* Enters a new mode */
void wsmode_begin()
{
	unsigned int i;
	/* Global init */
	if(!global_ready)
	{
		/* Global add */
		for(i = 0;i < num_modes;i++)
		{
			/* Global? */
			if(mode_global[i])
			{
				/* Init */
				if(mode_init[i])
					mode_init[i]();
			}
		}
		global_ready = 1;
	}
	/* Are we changing modes? */
	if(!changing_modes)
		return; /* Then this operation should be ignored */
	/* Initialize the mode */
	wsaction_wipe();
	if(mode_init[current_mode])
		mode_init[current_mode]();
	/* Resume playing video */
	video_running = 1;
	/* No longer changing */
	changing_modes = 0;
}

/* Handles the current mode */
void wsmode_body()
{
	unsigned long i;
	/* Run global modes */
	for(i = 0;i < num_modes;i++)
	{
		/* Global? */
		if(mode_global[i])
		{
			/* Run */
			if(mode_main[i])
				mode_main[i]();
		}
	}
	/* Don't do anything on mode change */
	if(changing_modes)
		return; /* Then calling main might screw things up */
	/* Run the mode */
	if(mode_main[current_mode])
		mode_main[current_mode]();
}

/* Handles leaving the current mode */
void wsmode_end()
{
	/* If we're chaning modes, change */
	if(!changing_modes)
		return; /* No need to clean yet */
	/* Clean up */
	if(mode_exit[current_mode])
		mode_exit[current_mode]();
	/* Video will no longer play */
	video_running = 0;
	/* Target modes */
	current_mode = target_mode;
}

/* Cleans up global modes */
void wsmode_exit()
{
	unsigned long i;
	/* Clean up globals */
	for(i = 0;i < num_modes;i++)
	{
		/* is even global? */
		if(mode_global[i])
		{
			/* Exit mode */
			if(mode_exit[i])
				mode_exit[i]();
		}
	}
}

/* Add a mode to the mode system */
mode wsmode_create(void (*inif)(),void (*mainf)(),void (*exitf)())
{
	/* Check for init */
	if(num_modes == 0)
	{
		/* Initialize modes */
		memset(mode_init,0,sizeof(mode_init));
		memset(mode_main,0,sizeof(mode_main));
		memset(mode_exit,0,sizeof(mode_exit));
		/* Set global zero */
		memset(mode_global,0,sizeof(mode_global));
	}
	/* Check for modes exceeded */
	if(num_modes >= MAX_MODES)
	{
		wserror_show("Maximum modes exceeded.");
		return 0; /* Cannot make a new mode */
	}
	/* Create a new mode */
	mode_init[num_modes] = inif;
	mode_main[num_modes] = mainf;
	mode_exit[num_modes] = exitf;
	/* Increment new mode */
	num_modes++;
	/* Return newly created mode */
	return num_modes-1;
}

/* Adds a mode to the mode system that draws graphics of it's own */
mode wsmode_create_2d(void (*inif)(),void (*mainf)(),void (*exitf)(),void (*drawf)())
{
	/* Check for modes exceeded */
	if(num_modes >= MAX_MODES)
	{
		wserror_show("Maximum modes exceeded in 2D mode creation.");
		return 0; /* Cannot make a new mode */
	}
	/* Check for init */
	if(num_modes == 0)
		memset(mode_2d,0,sizeof(mode_2d));
	/* Create */
	mode_2d[num_modes] = drawf;
	/* Increment 2d */
	num_2d_modes++;
	/* Pass off */
	return wsmode_create(inif,mainf,exitf);
}

/* Changes modes during a mode */
void wsmode_change(mode new_mode)
{
	/* Check for invalid mode */
	if(new_mode >= num_modes)
	{
		wserror_show("Attempt to enter nonexistent mode.");
		return;
	}
	/* Check for global */
	if(mode_global[new_mode])
	{
		wserror_show("Attempt to enter global mode as current mode.");
		return;
	}
	/* Set target and flag change */
	target_mode = new_mode;
	changing_modes = 1;
	/* Announce */
	wslog_string("[MODE] Changed to mode ");
	wslog_num(target_mode);
	wslog_return();
}

/* Resets the current mode */
void wsmode_reset()
{
	/* Just call this on current mode */
	wsmode_change(current_mode);
}

/* Sets the global value of a mode */
void wsmode_global(mode what,int val)
{
	/* Check for current */
	if(what == current_mode)
	{
		wserror_show("Cannot make a mode global if it is the current mode.");
		return;
	}
	/* If prerun */
	if(!engine_running)
	{
		mode_global[what] = val;
		return;
	}
	/* If adding */
	if(val)
	{
		/* Attempt initialize */
		if(mode_init[what])
			mode_init[what]();
		/* Set global */
		mode_global[what] = 1;
		/* Announce */
		wslog_string("[MODE] Set mode ");
		wslog_num(what);
		wslog_string(" to global.\n");
		return;
	}
	/* Detract */
	if(mode_exit[what])
		mode_exit[what]();
	/* Release global */
	mode_global[what] = 0;
	/* Announce */
	wslog_string("[MODE] Set mode ");
	wslog_num(what);
	wslog_string(" to inactive.\n");
}
