/* Input System */

/* Defines */
#ifndef MAX_PLAYER
#define MAX_PLAYER 8
#endif

/* Event */
SDL_Event inputevent;

/* Generic movement force */
unsigned char *keyboard_state;
int number_of_keyboard_keys;
int keyboard_force_x = 0;
int keyboard_force_y = 0;
double key_force_x = 0;
double key_force_y = 0;

/* Joysticks */
SDL_Joystick *joyref[MAX_PLAYER-1];

/* Player to player input states */
int input_method[MAX_PLAYER];
int player_a_button[MAX_PLAYER];
int player_b_button[MAX_PLAYER];
double player_x_force[MAX_PLAYER];
double player_y_force[MAX_PLAYER];

/* Prepare input systems */
void wsinput_init()
{
	int i;
	/* Memset all */
	memset(player_a_button,0,sizeof(player_a_button));
	memset(player_b_button,0,sizeof(player_b_button));
	/* Set default controls */
	for(i = 0;i < MAX_PLAYER;i++)
		input_method[i] = INPUT_NONE;
	/* Detections */
	for(i = 0;i < wsinput_num();i++)
	{
		/* Assign */
		input_method[i] = i;
		/* Is controller, then open */
		if(i > 0)
		{
			joyref[i-1] = SDL_JoystickOpen(i-1);
			if(!joyref[i-1])
				wserror_show("Joystick opening error");
		}
		/* Log */
		wslog_string("[INPUT] ");
		wslog_string(wsinput_name(i));
		/* With buttons */
		if(i > 0)
		{
			wslog_string(" (Buttons: ");
			wslog_num(SDL_JoystickNumButtons(joyref[i-1]));
			wslog_string(") ");
		}
		wslog_string(" is ready\n");
	}
	/* All ready */
	wslog_string("[INPUT] Systems are go\n");
}

/* Close input systems */
void wsinput_exit()
{
	int i;
	/* Wipe joys */
	for(i = 0;i < wsinput_num()-1;i++)
		SDL_JoystickClose(joyref[i]);
}

/* Gather all the input from everything and assign to players */
void wsplayer_get()
{
	int i;
	double getx,gety;
	/* All players */
	for(i = 0;i < MAX_PLAYER;i++)
	{
		/* Reset previous stats */
		player_a_button[i] = 0;
		player_b_button[i] = 0;
		player_x_force[i] = 0;
		player_y_force[i] = 0;
		/* Check for no controller cancel */
		if(input_method[i] != INPUT_NONE)
		{
			/* Keyboard operation */
			if(input_method[i] == INPUT_KEYBOARD)
			{
				/* Check for Z key press */
				if(keyboard_state[SDLK_z])
					player_a_button[i] = 1;
				/* Check for X key press */
				if(keyboard_state[SDLK_x])
					player_b_button[i] = 1;
				/* Check for directions */
				if(keyboard_state[SDLK_UP])
					player_y_force[i] = -1;
				if(keyboard_state[SDLK_DOWN])
					player_y_force[i] = 1;
				if(keyboard_state[SDLK_LEFT])
					player_x_force[i] = -1;
				if(keyboard_state[SDLK_RIGHT])
					player_x_force[i] = 1;
			}
			else
			{
				/* Check for (1) button press */
				if(SDL_JoystickGetButton(joyref[input_method[i]-1],0))
					player_a_button[i] = 1;
				/* Check for (2) button press */
				if(SDL_JoystickGetButton(joyref[input_method[i]-1],1))
					player_b_button[i] = 1;
				/* Check for no hats and use joystick */
				if(SDL_JoystickGetHat(joyref[input_method[i]-1],0) == 0)
				{
					/* Joysticks used */
					getx = SDL_JoystickGetAxis(joyref[input_method[i]-1],0);
					gety = SDL_JoystickGetAxis(joyref[input_method[i]-1],1);
					/* Normalize */
					getx /= 32768;
					gety /= 32768;
					/* Apply */
					player_y_force[i] = gety;
					player_x_force[i] = getx;
				}
				else
				{
					/* Hats override directions */
					if(SDL_JoystickGetHat(joyref[input_method[i]-1],0)&SDL_HAT_UP)
						player_y_force[i] = -1;
					if(SDL_JoystickGetHat(joyref[input_method[i]-1],0)&SDL_HAT_DOWN)
						player_y_force[i] = 1;
					if(SDL_JoystickGetHat(joyref[input_method[i]-1],0)&SDL_HAT_LEFT)
						player_x_force[i] = -1;
					if(SDL_JoystickGetHat(joyref[input_method[i]-1],0)&SDL_HAT_RIGHT)
						player_x_force[i] = 1;
				}
			}
		}
	}
}

/* Gather all the input from everything and optionally signal an engine halt */
void wsinput_get()
{
	/* Check for engine halt on quit */
	while(SDL_PollEvent(&inputevent))
	{
		/* All stop on exit */
		if(inputevent.type == SDL_QUIT)
			engine_running = 0;
		if(inputevent.type == SDL_KEYDOWN && inputevent.key.keysym.sym == SDLK_ESCAPE)
				engine_running = 0;
	}
	/* Keyboard force */
	keyboard_state = SDL_GetKeyState(&number_of_keyboard_keys);
	keyboard_force_y = 0;
	keyboard_force_x = 0;
	/* Press */
	if(keyboard_state[SDLK_UP])
		keyboard_force_y = -1;
	if(keyboard_state[SDLK_DOWN])
		keyboard_force_y = 1;
	if(keyboard_state[SDLK_LEFT])
		keyboard_force_x = -1;
	if(keyboard_state[SDLK_RIGHT])
		keyboard_force_x = 1;
	/* Update forces */
	key_force_x = (double)keyboard_force_x;
	key_force_y = (double)keyboard_force_y;
	/* Update */
	SDL_JoystickUpdate();
	/* Now players */
	wsplayer_get();
}

/* Returns the force of X movement */
double wsinput_move_x()
{
	return key_force_x;
}

/* Returns the force of Y movement */
double wsinput_move_y()
{
	return key_force_y;
}

/* Returns the number of game controllers connected */
int wsinput_num()
{
	return SDL_NumJoysticks()+1;
}

/* Returns the name of the joystick with a number */
char *wsinput_name(int index)
{
	if(index == 0)
		return "Keyboard";
	return (char*)SDL_JoystickName(index-1);
}

/* Returns the name of the input device used by player # */
char *wsplayer_name(int index)
{
	if(input_method[index] == INPUT_NONE)
		return "No controller plugged in";
	return wsinput_name(input_method[index]);
}

/* Returns the player's A button status */
int wsplayer_a(int index)
{
	return player_a_button[index];
}

/* Returns the player's B button status */
int wsplayer_b(int index)
{
	return player_b_button[index];
}

/* Returns the player's X force */
double wsplayer_x(int index)
{
	return player_x_force[index];
}

/* Returns the player's Y force */
double wsplayer_y(int index)
{
	return player_y_force[index];
}

/* Returns the input method for the player */
int wsplayer_method(int index)
{
	return input_method[index];
}
