/*
	WeaponSoul 2D Image System
	written by WolfCoder (2009)

	Uses the OpenGL Graphics Library

	Images are set in a pool based system where even if you never release
	a pack of images, it is accounted for.
	"Loading" an image is actually only getting it's index.
	It is only actually loaded from a pack of images.
*/

/* Includes */
#include "weaponsoul.h"
#include <GL/gl.h> /* OpenGL */
#include <memory.h> /* Memory */

/* Globals */
FILE *wipfile;
unsigned long filetag = 0x32504957;
unsigned long num_images = 0; /* Number of images loaded */
unsigned long video_memory = 0; /* Amount of video memory used up */
unsigned long current_image = 0; /* Current image next to be loaded */
unsigned char *data; /* Bitmap data */
unsigned long dword; /* Temp 32 bit reading variable */

/* Image Data */
folder image_source[MAX_IMAGES]; /* Image source pack name */
name image_name[MAX_IMAGES]; /* Name of image */
int image_use[MAX_IMAGES]; /* Is this image in use? */
int image_width[MAX_IMAGES]; /* Width of the image frame */
int image_height[MAX_IMAGES]; /* Height of the image frame */
int image_frames[MAX_IMAGES]; /* Number of frames in the image */
GLuint *image_texture[MAX_IMAGES]; /* Texture data for image */

/* Image Parameters */
int next_frame; /* Next frame to look like */
double sx,sy; /* Next scale factor */
double r; /* Next rotation */
double alpha; /* Next alpha */
double lred,lgreen,lblue; /* Next color */
int subtract; /* Do a color subtraction instead */

/* Returns the current position for loading of images */
unsigned long wsimage_current()
{
	return current_image;
}

/* Returns the number of images loaded in memory */
unsigned long wsimage_num()
{
	return num_images;
}

/* Initialize the image system */
void wsimage_init()
{
	/* Initialize usage data */
	memset(image_use,0,sizeof(image_use));
	/* Default params */
	next_frame = 0;
	sx = 1;
	sy = 1;
	r = 0;
	alpha = 1;
}

/* Add a single image to the pool */
void wsimage_add(char *filename)
{
	unsigned long fn,fw,fh;
	unsigned long i;
	/* Check for out of room */
	if(num_images >= MAX_IMAGES)
	{
		wserror_show("Out of image room.");
		return;
	}
	/* Seek */
	while(1)
	{
		if(image_use[current_image])
		{
			/* Advance */
			current_image++;
			if(current_image >= MAX_IMAGES)
				current_image = 0;
		}
		else
		{
			/* Found! */
			break;
		}
	}
	/* Found! Add the image */
	image_use[current_image] = 1;
	wsfolder_set(image_source[current_image],filename);
	wsname_read(image_name[current_image],wipfile);
	/* Load number, width, height of each frame */
	fread(&fn,4,1,wipfile);
	fread(&fw,4,1,wipfile);
	fread(&fh,4,1,wipfile);
	/* Allocate OpenGL pointers */
	image_texture[current_image] = (GLuint*)wsalloc(sizeof(GLuint)*fn);
	glGenTextures(fn,image_texture[current_image]);
	/* Offload all the frames to the video card */
	for(i = 0;i < fn;i++)
	{
		/* Allocate */
		data = (unsigned char*)wsalloc(fw*fh*4);
		/* Read */
		fread(data,4,fw*fh,wipfile);
		/* Offload */
		glBindTexture(GL_TEXTURE_2D,image_texture[current_image][i]);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
		glTexImage2D(GL_TEXTURE_2D,0,4,fw,fh,0,GL_RGBA,GL_UNSIGNED_BYTE,data);
		/* Deallocate */
		wsalloc_free(data);
	}
	/* Set width and height */
	image_width[current_image] = fw;
	image_height[current_image] = fh;
	image_frames[current_image] = fn;
	/* Announce */
	wslog_string("\tImage ");
	wslog_name(image_name[current_image]);
	wslog_string(" loaded (");
	wslog_num(fw);
	wslog_string(",");
	wslog_num(fh);
	wslog_string(") - ");
	wslog_num(fn);
	wslog_string(" frames.\n");
	/* Tally and increment */
	video_memory += fw*fh*4;
	num_images++;
	current_image++;
}

/* Load a pack of images into the engine */
void wsimage_load(char *filename)
{
	unsigned int i;
	unsigned long num_files;
	name pack_name;
	/* Open the file */
#ifndef WIN32
	wipfile = fopen(filename,"rb");
#endif
#ifdef WIN32
	fopen_s(&wipfile,filename,"rb");
#endif
	/* Check the file tag */
	fread(&dword,4,1,wipfile);
	if(dword != filetag)
	{
		wserror_show("Corrupted/Invalid .WIP was loaded.");
		fclose(wipfile);
		return;
	}
	/* Read number of files */
	fread(&num_files,4,1,wipfile);
	/* Read name */
	wsname_read(pack_name,wipfile);
	/* Add all images */
	for(i = 0;i < num_files;i++)
		wsimage_add(filename);
	/* Close the file */
	fclose(wipfile);
	/* Announce */
	wslog_string("[IMAGE] Loaded ");
	wslog_num(num_files);
	wslog_string(" images from '");
	wslog_string(filename);
	wslog_string("' (");
	wslog_num(num_images);
	wslog_string(" total).\n");
	wslog_string("\t");
	wslog_num((video_memory/1024));
	wslog_string("K total VRAM in use.\n");
}

/* Release an image */
void wsimage_delete(image what)
{
	/* Valid? */
	if(!image_use[what])
		return;
	/* Not in use */
	image_use[what] = 0;
	/* Free */
	glDeleteTextures(image_frames[what],image_texture[what]);
	/* Delete pointers */
	wsalloc_free(image_texture[what]);
	/* Remove from total */
	num_images--;
	video_memory -= image_width[what]*image_height[what]*4;
	/* Announce */
	wslog_string("\tImage ");
	wslog_name(image_name[what]);
	wslog_string(" deleted.\n");
}

/* Free a pack of images from the engine */
void wsimage_free(char *filename)
{
	int i;
	int num_del;
	folder compare;
	/* Prepare the compare folder */
	wsfolder_set(compare,filename);
	/* Delete all matching source */
	num_del = 0;
	for(i = 0;i < MAX_IMAGES;i++)
	{
		/* Check and delete */
		if(wsfolder_equal(compare,image_source[i]))
		{
			/* Exists? */
			if(image_use[i])
			{
				num_del++;
				wsimage_delete(i);
			}
		}
	}
	/* Announce */
	wslog_string("[IMAGE] Deleted ");
	wslog_num(num_del);
	wslog_string(" images belonging to '");
	wslog_string(filename);
	wslog_string("' (");
	wslog_num(num_images);
	wslog_string(" total).\n");
	wslog_string("\t");
	wslog_num((video_memory/1024));
	wslog_string("K total VRAM in use.\n");
}

/* Get an image */
image wsimage_get(name what)
{
	unsigned long i;
	/* Find */
	for(i = 0;i < MAX_IMAGES;i++)
	{
		if(wsname_equal(what,image_name[i]))
			return i;
	}
	/* Error */
	wserror_show("Image does not exist.");
	return 0;
}

/* Changes the display frame for the immediate next image to be drawn */
void wsimage_frame(int frame)
{
	/* Set frame */
	next_frame = frame;
}

/* Changes the scaling for the next and only the next image */
void wsimage_scale(double scale_x,double scale_y)
{
	/* Set values */
	sx = scale_x;
	sy = scale_y;
}

/* Changes the rotation for the next image */
void wsimage_rotate(double rot)
{
	/* Set values */
	r = rot;
}

/* Changes the alpha for the next image */
void wsimage_alpha(double alph)
{
	/* Set values */
	alpha = alph;
}

/* Changes color filter for next image */
void wsimage_color(double red,double green,double blue)
{
	/* Set color */
	lred = red;
	lblue = blue;
	lgreen = green;
}

/* Specifies if the color should be added or subtracted */
void wsimage_subtract(int sub)
{
	subtract = sub;
}

/* Draws an image in it's current parameters at x,y */
void wsimage_draw(image what,int x,int y)
{
	GLfloat gx,gy;
	/* Set position */
	gx = (GLfloat)(x+image_width[what]/2);
	gy = (GLfloat)(y+image_height[what]/2);
	/* Position */
	glPushMatrix();
	glTranslatef(gx,gy,0);
	/* Rotate */
	if(r != 0)
		glRotatef((GLfloat)r,0,0,1);
	/* Scale */
	if(sx != 1 || sy != 1)
		glScalef((GLfloat)sx,(GLfloat)sy,0);
	/* Select the texture */
	if(next_frame >= image_frames[what])
		next_frame = 0;
	glBindTexture(GL_TEXTURE_2D,image_texture[what][next_frame]);
	/* Set blends */
	if(lred != 0 || lgreen != 0 || lblue != 0)
	{
		/* Apply a color blend */
		glColor4f((GLfloat)lred,(GLfloat)lgreen,(GLfloat)lblue,(GLfloat)alpha);
		if(subtract)
			glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
		else
			glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_ADD);
	}
	else
	{
		/* Just alpha */
		if(alpha < 1)
		{
			glColor4f(1,1,1,(GLfloat)alpha);
			glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
		}
	}
	/* Draw the quad */
	glBegin(GL_QUADS);
	glTexCoord2f(0,0);
	glVertex3f((GLfloat)-image_width[what]/2,(GLfloat)-image_height[what]/2,0);
	glTexCoord2f(0,1);
	glVertex3f((GLfloat)-image_width[what]/2,(GLfloat)image_height[what]/2,0);
	glTexCoord2f(1,1);
	glVertex3f((GLfloat)image_width[what]/2,(GLfloat)image_height[what]/2,0);
	glTexCoord2f(1,0);
	glVertex3f((GLfloat)image_width[what]/2,(GLfloat)-image_height[what]/2,0);
	/* End passes */
	glEnd();
	glPopMatrix();
	/* End alpha */
	if(lred != 0.5 || lgreen != 0.5 || lblue != 0.5)
	{
		glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
		glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);
	}
	else
	{
		if(alpha < 1)
			glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);
	}
	/* Reset parameters */
	next_frame = 0;
	sx = 1;
	sy = 1;
	r = 0;
	alpha = 1;
	lred = 0;
	lblue = 0;
	lgreen = 0;
	subtract = 0;
}

/* Gets dimensions of an image */
int wsimage_width(image what)
{
	return image_width[what];
}
int wsimage_height(image what)
{
	return image_height[what];
}

/* Gets the number of frames in an image */
int wsimage_frames(image what)
{
	return image_frames[what];
}

/* End the image system */
void wsimage_exit()
{
	unsigned long i;
	/* Delete all texture data */
	for(i = 0;i < MAX_IMAGES;i++)
		wsimage_delete(i);
}
