#ifndef WSACTION_H
#define WSACTION_H

/* Action types */
#define ENTITY_SPIRIT 0
#define ENTITY_SPRITE 1
#define ENTITY_TILEMAP 2

/* Max field */
#define INIT_ENTITIES 256
#ifndef MAX_DATA
#define MAX_DATA 256
#endif
#define MAX_ACTION 256

#define ENTITY_BRIGHT 0
#define ENTITY_DARK 1

/* Event types */
#define EVENT_IMPACT 0

/* Typedef */
typedef unsigned long action;
typedef unsigned long response;

/* Struct */
typedef struct
{
	/* Orientation */
	double x,y,z,p,t,r,sx,sy,sz;
	double dx,dy,dz,dp,dt,dr,dsx,dsy,dsz;
	/* Offset */
	double ox,oy,oz;
	/* Movement */
	double vx,vy,vz;
	/* Basic size */
	double h,l,w;
	/* Transparency */
	double a;
	/* Color */
	double red,green,blue;
	/* Color mode */
	int colormode;
	/* Frame */
	int frame;
	/* Skills */
	unsigned char data[MAX_DATA];
	void *payload;
	int payloadable;
	/* Action */
	action (*behave)();
	/* Event */
	response (*handler)(int);
	/* Alive value */
	int alive;
	/* Initializing */
	int init;
	/* Passable? */
	int passable;
	/* Visible? */
	int invisible;
	/* Frame base and additions */
	int auto_animation;
	int auto_animation_width;
	double frame_base;
	double frame_add;
	double frame_dist;
	double frame_dir;
	/* Sprite body */
	image s_body;
	/* Layer selection */
	int layer;
	/* Type */
	int type;
}entity_data;

/* Entity */
typedef entity_data *entity;

/* Externs */
extern entity me,you,last;
extern void *my_data;

/* For use by engine designer */
extern void wsaction_init();
extern void wsaction_exit();
extern void wsaction_main();
extern entity wsaction_data(unsigned long who);
extern unsigned long wsaction_bound();

/*
	Creates a new action and sets it in motion.

	The pointer to action will be executed alongside
	the game. It will remain until it is deleted, or
	when a wsaction_wipe() call is made, removing all entities.
*/
extern void wsaction_create(action (*beh)());

/*
	Returns the number of entities existing.
*/
extern unsigned long wsaction_num();

/*
	Deletes the entity given.

	Remember, simply returning 0 in an entity action
	will ALSO delete it.
*/
extern void wsaction_delete(entity who);

/*
	Deletes all entities, and relases
	excessive memory used by them.
	This might also increase frame rate.
*/
extern void wsaction_wipe();

/*
	Returns a nonzero value if the
	pointer to entity is alive or
	even exists at all alive.
*/
extern int wsaction_alive(entity who);

/*
	Moves the entity, performing any collision checks
	Returns a non zero number if the movement was successful
*/
extern int wsaction_move(entity who,double x,double y,double z);

/*
	Changes the entity's event handler
*/
extern void wsaction_event(entity who,response (*newevn)(int));

/*
	Scales the entity
*/
extern void wsaction_scale(entity who,double sx,double sy,double sz);

/*
	Uniform scale the entity
*/
extern void wsaction_size(entity who,double sf);

/*
	Updates the orientation of entity
*/
extern void wsaction_angle(entity who,double pan,double tilt,double roll);

/*
	Rotates the entity
*/
extern void wsaction_rotate(entity who,double pan,double tilt,double roll);

/*
	Sets entity transparency
*/
extern void wsaction_alpha(entity who,double alpha);

/*
	Sets entity color filter
*/
extern void wsaction_color(entity who,double red,double green,double blue,int sub);

/*
	Makes an entity visible or invisible
*/
extern void wsaction_invisible(entity who,int invis);

/*
	Makes an entity passable or impassable
*/
extern void wsaction_passable(entity who,int pass);

/*
	Attaches a payload of the size given to the entity specified
	A payload is an arbitrary length of data that is allocated when this function is called
	This function will fail if a payload has already been attached
	Payloads are automatically freed when the entity that it is attached to is deleted
	Payloads also fail on entities that can't have payloads, such as tilemaps
*/
extern void *wsaction_payload(entity who,size_t size);

/*
	Changes the collision hull for the entity
*/
extern void wsaction_hull(entity who,double w,double l,double h);

/*
	Returns true if the point given is inside any solid entities
*/
extern int wsaction_contains(double x,double y,double z);

/*
	Returns the bytes held in memory for use with Entity system
*/
extern unsigned long wsaction_memory();

#endif
