/*
	WeaponSoul Action System
	puts life of behavior into things
	written by WolfCoder (2009)
*/

/* Includes */
#include "weaponsoul.h"
#include <memory.h>

/* Extern Pointers */
entity me,you,last;
void *my_data;

/* Globals */
entity_data *edata;
unsigned long entity_size; /* Allocated memory for entities */
unsigned long current_entity; /* Current insertion pointer */
unsigned long num_entities; /* Number of active entities */
unsigned long upper_bound; /* It's verified that no entities exist past this point */
unsigned long memory_held; /* Total amount of memory held by system */
unsigned long memory_used; /* Total amount of memory held by active entities */
int memory_block = 1; /* Blocks memory access */

/* Initializer */
void wsaction_init()
{
	unsigned long i;
	/* Initial values */
	current_entity = 0;
	num_entities = 0;
	entity_size = INIT_ENTITIES;
	upper_bound = 0;
	/* Allocate room */
	edata = (entity_data*)wsalloc(sizeof(entity_data)*INIT_ENTITIES);
	/* Set new memory held */
	memory_held = sizeof(entity_data)*INIT_ENTITIES;
	/* Initialize as inactive */
	for(i = 0;i < entity_size;i++)
		edata[i].alive = 0;
	/* Unblock */
	memory_block = 0;
}

/* Get the upper bound */
unsigned long wsaction_bound()
{
	return upper_bound;
}

/* Create an entity and set it in motion */
void wsaction_create(action (*beh)())
{
	entity newdata;
	/* Check error */
	if(!beh)
	{
		wserror_show("Can't create an entity without an action.");
		return;
	}
	/* Scan for an empty spot */
	if(num_entities < entity_size)
	{
		/* Seek */
		while(current_entity < entity_size)
		{
			if(!edata[current_entity].alive)
			{
				/* Destroy the old data */
				memset(&edata[current_entity],0,sizeof(entity_data));
				/* Set alive */
				edata[current_entity].alive = 1;
				edata[current_entity].init = 1;
				edata[current_entity].sx = 1;
				edata[current_entity].sy = 1;
				edata[current_entity].sz = 1;
				edata[current_entity].a = 1;
				edata[current_entity].auto_animation = 0;
				edata[current_entity].handler = NULL;
				edata[current_entity].red = 0;
				edata[current_entity].blue = 0;
				edata[current_entity].green = 0;
				edata[current_entity].colormode = ENTITY_BRIGHT;
				edata[current_entity].payloadable = 1;
				edata[current_entity].payload = NULL;
				/* Set data */
				memset(&edata[current_entity].data,0,sizeof(unsigned char)*MAX_DATA);
				/* Set type */
				edata[current_entity].type = ENTITY_SPIRIT;
				/* Set action */
				edata[current_entity].behave = beh;
				/* Set last */
				last = &edata[current_entity];
				/* Increase used bytes */
				num_entities++;
				/* Increment current */
				current_entity++;
				/* Check for update */
				if(current_entity > upper_bound)
					upper_bound = current_entity;
				/* Add used memory */
				memory_used += sizeof(entity_data);
				return;
			}
			/* Not found, seek */
			current_entity++;
		}
		/* Serious error */
		wserror_show("Dangerous unknown error occur in entity creation.");
		return;
	}
	/* Block memory */
	memory_block = 1;
	/* Out of entity memory, expand and then add right after */
	newdata = (entity_data*)wsalloc(sizeof(entity_data)*entity_size*2); /* Allocate */
	/* Fatal error on memory loss */
	if(!newdata)
	{
		wserror_show("Out of memory for creating entities.");
		return;
	}
	/* Copy */
#ifndef WIN32
	memcpy(newdata,edata,sizeof(entity_data)*entity_size);
#endif
#ifdef WIN32
	memcpy_s(newdata,sizeof(entity_data)*entity_size*2,edata,sizeof(entity_data)*entity_size);
#endif
	/* Now delete the old edata */
	wsalloc_free(edata);
	/* Reassign */
	edata = newdata;
	/* Set new memory held */
	memory_held = sizeof(entity_data)*entity_size*2;
	/* Set insert */
	current_entity = entity_size;
	entity_size *= 2;
	/* Unblock */
	memory_block = 0;
	/* Recall with current_entity set at the new position */
	wsaction_create(beh);
}

/* Delete an entity */
void wsaction_delete(entity who)
{
	/* Already dead */
	if(!who->alive)
		return;
	/* Set dead */
	edata->alive = 0;
	num_entities--;
	/* Erase payload */
	wsalloc_free(who->payload);
	/* Set sprite dead */
	if(edata->type == ENTITY_SPRITE)
		wssprite_reduce();
}

/* Handle all actions */
void wsaction_main()
{
	unsigned long i;
	/* Nullify pointers */
	me = NULL;
	you = NULL;
	my_data = NULL;
	/* Handle all entities */
	for(i = 0;i < upper_bound;i++)
	{
		/* Alive? */
		if(edata[i].alive)
		{
			/* Handle */
			me = &edata[i];
			my_data = &edata[i].data;
			you = NULL;
			/* Execute */
			if(!me->behave())
			{
				wsaction_delete(wsaction_data(i));
			}
			else
			{
				/* Check for init */
				if(me->init)
					me->init = 0;
				/* Finalize drawing positions */
				me->dx = me->x;
				me->dy = me->y;
				me->dz = me->z;
				me->dp = me->p;
				me->dt = me->t;
				me->dr = me->r;
				me->dsx = me->sx;
				me->dsy = me->sy;
				me->dsz = me->sz;
				/* Auto animations */
				if(me->auto_animation)
					me->frame = (int)(me->frame_base+me->frame_add);
				/* Auto moves */
				wsaction_move(me,me->vx,me->vy,me->vz);
			}
		}
	}
	/* Reset addition counter */
	current_entity = 0;
}

/* Clean up objects */
void wsaction_exit()
{
	/* Wipe */
	wsaction_wipe();
	/* Free */
	wsalloc_free(edata);
}

/* Wipe everything down */
void wsaction_wipe()
{
	int i;
	/* Remove all payloads */
	for(i = 0;i < upper_bound;i++)
	{
		if(edata[i].alive)
			wsalloc_free(edata[i].payload);
	}
	/* KILL ALL DATA */
	wsalloc_free(edata);
	/* Reset */
	wsaction_init();
}

/* Returns the amount of entities alive */
unsigned long wsaction_num()
{
	return num_entities;
}

/* Returns if the following entity is actually alive */
int wsaction_alive(entity who)
{
	/* Block temporal memory */
	if(memory_block)
		return 0;
	/* Normal drawing */
	return who->alive;
}

/* Returns the pointer to direct data of an entity */
entity wsaction_data(unsigned long who)
{
	/* Exists? return */
	if(who < upper_bound)
		return &edata[who];
	/* Nonexistent */
	return NULL;
}

/* Does cross checking for collisions */
int wsaction_collision(entity who)
{
	int i;
	/* Examine everything */
	for(i = 0;i < upper_bound;i++)
	{
		/* Must be existing */
		if(edata[i].alive)
		{
			/* Must be solid */
			if(!edata[i].passable)
			{
				/* Must not be self */
				if(&edata[i] != who)
				{
					/* Check */
					if(wshull_check(who,&edata[i]))
					{
						/* Trigger events */
						/* Return success */
						return 1;
					}
				}
			}
		}
	}
	/* No detection */
	return 0;
}

/* Moves an entity, performing any bounding box checks */
int wsaction_move(entity who,double x,double y,double z)
{
	double distx,disty,distz;
	/* Check no move */
	if(x == 0 && y == 0 && z == 0)
		return 0;
	/* Measure distance */
	distx = x*TIME_STEP;
	disty = y*TIME_STEP;
	distz = z*TIME_STEP;
	/* Move */
	me->x += distx;
	me->y += disty;
	me->z += distz;
	/* Passable is not included in detection */
	if(who->passable)
		return 1;
	/* Detect */
	if(!wsaction_collision(who))
		return 1;
	/* Revert */
	me->x -= distx;
	me->y -= disty;
	me->z -= distz;
	/* Failure to move */
	return 0;
}

/* Changes event */
void wsaction_event(entity who,response (*newevn)(int))
{
	who->handler = newevn;
}

/* Scales entity */
void wsaction_scale(entity who,double sx,double sy,double sz)
{
	who->sx = sx;
	who->sy = sy;
	who->sz = sz;
}

/* Scales entity uniformily */
void wsaction_size(entity who,double sf)
{
	wsaction_scale(who,sf,sf,sf);
}

/* Points the entity in a direction */
void wsaction_angle(entity who,double pan,double tilt,double roll)
{
	who->p = pan;
	who->t = tilt;
	who->r = roll;
}

/* Rotates the entity by an amount */
void wsaction_rotate(entity who,double pan,double tilt,double roll)
{
	/* Measure distance */
	pan *= TIME_STEP;
	tilt *= TIME_STEP;
	roll *= TIME_STEP;
	/* Apply */
	wsaction_angle(who,who->p+pan,who->t+tilt,who->r+roll);
}

/* Sets the alpha for the entity */
void wsaction_alpha(entity who,double alpha)
{
	who->a = alpha;
}

/* Sets the color filter for the entity */
void wsaction_color(entity who,double red,double green,double blue,int sub)
{
	who->red = red;
	who->green = green;
	who->blue = blue;
	who->colormode = sub;
}

/* Makes an entity passable and excludes it from collision detection */
void wsaction_passable(entity who,int pass)
{
	who->passable = pass;
}

/* Makes an entity invisible */
void wsaction_invisible(entity who,int invis)
{
	who->invisible = invis;
}

/* Attaches a payload to the entity of size given and returns the pointer to that data */
void *wsaction_payload(entity who,size_t size)
{
	/* Cannot payload an unpayloadable entity */
	if(!who->payloadable)
	{
		wserror_show("Payload attach attempt on an entity that can't have payloads.");
		return NULL;
	}
	/* Create and attach */
	who->payload = wsalloc(size);
	who->payloadable = 0;
	/* Return result */
	return who->payload;
}

/* Changes the actions hull */
void wsaction_hull(entity who,double w,double l,double h)
{
	/* Set hull */
	who->w = w;
	who->l = l;
	who->h = h;
}

/* Checks a point for solids */
int wsaction_contains(double x,double y,double z)
{
	int i;
	/* Run through all */
	for(i = 0;i < upper_bound;i++)
	{
		/* Must be alive */
		if(edata[i].alive)
		{
			/* Must be solid */
			if(!edata[i].passable)
			{
				/* Check */
				if(wshull_point(x,y,z,&edata[i]))
					return 1;
			}
		}
	}
	/* Came up with nothing */
	return 0;
}

/* Gets memory used */
unsigned long wsaction_memory()
{
	return memory_held;
}
