#ifndef WEAPONSOUL_H
#define WEAPONSOUL_H

/*
	This is a macro that contains a value you can multiply your
	speeds with to get frame-independant movement. Many moving
	instructions already use this macro and you might not need
	to use it yourself.
*/
#define TIME_STEP time_step()

/*
	Max field numbers
*/
#define MAX_MODES 64
#define MAX_NAME 32
#define MAX_FOLDER 256

/* Values of input */
#define INPUT_NONE -1
#define INPUT_KEYBOARD 0

/*
	Current version of WeaponSoul.
*/
#define WS_VERSION 4

/*
	Including this file will include windows for windows systems
*/
#ifdef WIN32
#include <windows.h>
#endif

/*
	Including this file will include stdio and lib
*/
#include <stdio.h>
#include <stdlib.h>

/*
	Call ws_main to provide the main function for the program
*/
#ifdef WIN32
#define ws_main int WINAPI WinMain(HINSTANCE hinstance,HINSTANCE hprevinstance,PSTR szcmdline,int icmdshow)
#endif
#ifndef WIN32
#define ws_main int main(int argn,char *argv[])
#endif

/*
	Types
*/
typedef unsigned long mode;
typedef char name[MAX_NAME];
typedef char folder[MAX_FOLDER];

/* Internal use */
extern void wscamera_draw_copy();
extern double wscamera_draw_x();
extern double wscamera_draw_y();
extern double wscamera_draw_z();

/*
	Starts the engine. After this call is made,
	calls after it will only be made after the entire
	engine has finished running.

	Set all engine settings before you call this function
	for the game to be how you want it.
*/
extern void ws_start();

/*
	Displays an error.
*/
extern void wserror_show(char *msg);

/*
	Creates a new mode for the game.

	The first mode created is the initial mode the game
	will start in.

	You can change modes in one mode to go from one
	mode to another.
*/
extern mode wsmode_create(void (*inif)(),void (*mainf)(),void (*exitf)());

/*
	Does the same thing as wsmode_create
	but the drawf function draws in 2D using
	methods of it's own.

	For use with things like HUD
*/
extern mode wsmode_create_2d(void (*inif)(),void (*mainf)(),void (*exitf)(),void (*drawf)());

/*
	Changes the mode to a new one.

	If set to change into a mode thats already the current mode,
	a resetting effect will happen.
*/
extern void wsmode_change(mode new_mode);

/*
	Resets the current mode.

	This has the same effect as wsmode_change(current_mode);
*/
extern void wsmode_reset();

/*
	Makes a mode global so it is always running, like
	a background process of sorts or a persistent process
	like a HUD mode. The idea of being "MODAL" is discarded and
	it becomes a process.
*/
extern void wsmode_global(mode what,int val);

/*
	Allocates some memory, initialized to 0.
	Also keeps track of each allocation.

	Parameters are the same as for malloc
*/
extern void *wsalloc(size_t len);

/*
	Frees some memory, and checks for matching allocations.
*/
extern void wsalloc_free(void *what);

/*
	Stops the engine and quits the game.

	This is a thread-safe function used to tell the
	engine to stop all operatins. Stopping does not happen
	immediatly (being thread-safe) so be sure to not assume
	the program will not execute anything after this function call.
*/
extern void ws_quit();

/*
	Sets a name to the contents of the null terminated string.

	If the string is too long, up to MAX_NAME characters will
	actually be copied.
*/
extern void wsname_set(name what,char *str);

/*
	Reads a name from the file given.
	If the string is smaller than the maximum length of the
	fixed-length string, only the bytes (including the zero)
	that make up the string are read and it stops reading.
*/
extern void wsname_read(name what,FILE *from);

/*
	Returns a non-zero number if two names are the same.
*/
extern int wsname_equal(name a,name b);

/*
	These work the same as the name functions, but
	they apply to folders which are fixed strings
	of greater size to hold file paths.
*/
extern void wsfolder_set(folder what,char *str);
extern int wsfolder_equal(folder a,folder b);

/*
	Output a name in the log
*/
extern void wslog_name(name what);

/*
	Hit return in the log
*/
extern void wslog_return();

/*
	Output a string in the log
*/
extern void wslog_string(char *what);

/*
	Output a number in the log
*/
extern void wslog_num(int what);

/*
	Output a real number in the log
	(floating point)
*/
extern void wslog_real(double what);

/*
	Changes the game resolution

	sw and sh are the width and height of the actual window or fullscreen resolutionm

	gw and gh are the internal resolution of the game.
*/
extern void wsvideo_resolution(int sw,int sh,int gw,int gh);

/*
	If a non-zero value is passed, the game will start
	in fullscreen mode.
	Otherwise, a windowed mode will be used.
*/
extern void wsvideo_mode(int new_mode);

/*
	Returns the width of the internal screen area.

	Useful for aligning things to the screen.
*/
extern int wsvideo_width();

/*
	Returns the height of the internal screen area.

	Useful for aligning things to the screen.
*/
extern int wsvideo_height();

/*
	Returns the duration of the last frame in milliseconds.
*/
extern unsigned long wstime_get();

/*
	Returns the value to multiply with all speeds in
	order to get frame independant movement speeds.
*/
extern double time_step();

/*
	Returns the current movement force direction.
	This is a range from 0 to 1 as not only the arrow keys
	can effect this value, but joysticks as well.
*/
extern double wsinput_move_x();

/*
	Returns the current movement force direction.
	This is a range from 0 to 1 as not only the arrow keys
	can effect this value, but joysticks as well.
*/
extern double wsinput_move_y();

/*
	Returns the number of game controllers
*/
extern int wsinput_num();

/*
	Returns the name of the input device indexed.
	Device 0 is always the mouse and keyboard.
*/
extern char *wsinput_name(int index);

/*
	Returns the name of the input device used by a player.
*/
extern char *wsplayer_name(int index);

/*
	Returns the status of the [A] button for the player given.
	The Z key and Button 1 are A buttons
*/
extern int wsplayer_a(int index);

/*
	Returns the status of the [B] button for the player given.
	The X key and Button 2 are B buttons
*/
extern int wsplayer_b(int index);

/*
	Returns the status of the X and Y forces for the player given.
	This is effected by POV/HATs, arrow keys and analog joysticks.
	Values will be between -1 and 1 with 0 being no input.
*/
extern double wsplayer_x(int index);
extern double wsplayer_y(int index);

/*
	Returns the input method of the player
*/
extern int wsplayer_method(int index);

/*
	Changes the title of the game window.
*/
extern void ws_title(char *new_title);

/*
	Moves the global camera.
	This effects tilemaps, sprites, and many other world objects
	to simulate a view
*/
extern void wscamera_move(double x,double y,double z);

/*
	Gets the global camera's current position
*/
extern double wscamera_x();
extern double wscamera_y();
extern double wscamera_z();

/*
	WeaponSoul Outer Modules
*/
#include "wsimage.h" /* 2D Image system */
#include "wsaction.h" /* Action process */
#include "wshull.h" /* Collision detection */
#include "wstext.h" /* Drawing of text */
#include "wssprite.h" /* Sprites */
#include "wstile.h" /* Tilemaps */

#endif
