// IMAGE
import java.awt.image.BufferedImage;

// GEOM
import java.awt.geom.AffineTransform;

// AWT
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Color;

/**
	WolfCoder Image
	<p>
	These are constructed using the ImagePack object.
	Methods exist in this object for easy construction by the ImagePack object.
	If you don't know what you're doing, use the ImagePack to return an array of these.

	@author WolfCoder
	@version 1
*/
public class WolfcoderImage
{
	private BufferedImage[] frame;
	private String name;
	private int width,height;
	/**
		Creates a new WolfCoder Image with a number of frames.
		Each image has a name.
		Width and Height are also required.

		@param width Width of each frame in pixels.
		@param height Height of each frame in pixels.
		@param frames Number of frames this image will have.
	*/
	public WolfcoderImage(String name,int width,int height,int frames)
	{
		// Set name
		this.name = name;
		// Set properties
		this.width = width;
		this.height = height;
		// Allocate
		frame = new BufferedImage[frames];
		for(int i = 0;i < frames;i++)
			frame[i] = new BufferedImage(width,height,BufferedImage.TYPE_4BYTE_ABGR);
	}
	/**
		Writes the 32-bit ARGB data to the frame given.

		@param data Array of bytes to be written as image data.
		@param index Frame to write to.
	*/
	public void write(int[] data,int index)
	{
		// Set data
		Graphics2D g2d = (Graphics2D)frame[index].getGraphics();
		for(int y = 0;y < height;y++)
		{
			for(int x = 0;x < width;x++)
			{
				g2d.setColor(new Color(data[(x+y*width)*4],data[1+(x+y*width)*4],data[2+(x+y*width)*4],data[3+(x+y*width)*4]));
				g2d.fillRect(x,y,1,1);
			}
		}
	}
	/**
		Gets the name of this WolfCoder Image.

		@return The name of this WolfCoder Image.
	*/
	public String getName()
	{
		return name;
	}
	/**
		Gets the width of each frame.

		@return The frame width.
	*/
	public int getWidth()
	{
		return width;
	}
	/**
		Gets the height of each frame.

		@return The frame height.
	*/
	public int getHeight()
	{
		return height;
	}
	/**
		Draws the image at (x,y) using the frame given.

		@param g Graphics context to draw into.
		@param x X location to draw the image.
		@param y Y location to draw the image.
		@param i Frame to draw with.
	*/
	public void draw(Graphics g,int x,int y,int i)
	{
		Graphics2D g2d = (Graphics2D)g;
		g2d.drawRenderedImage(frame[i],AffineTransform.getTranslateInstance(x,y));
	}
	/**
		Draws the image at (x,y) using the first frame.

		@param g Graphics context to draw into.
		@param x X location to draw the image.
		@param y Y location to draw the image.
	*/
	public void draw(Graphics g,int x,int y)
	{
		draw(g,x,y,0);
	}
	/**
		Returns a BufferedImage that corresponds to the frame given.

		@param index Frame to return as a BufferedImage.
		@return The BufferedImage of the frame specified.
	*/
	public BufferedImage getImage(int index)
	{
		return frame[index];
	}
	/**
		Returns the string representation of this WolfcoderImage.

		@return The string representation of this WolfcoderImage.
	*/
	public String toString()
	{
		return new String("WolfCoderIMG "+name+" w:"+frame[0].getWidth()+" h:"+frame[0].getHeight()+" n:"+frame.length);
	}
}
