// IO
import java.io.File;
import java.io.FileInputStream;
import java.io.DataInputStream;

/**
	WolfCoder Image Pack
	<p>
	This object represents an image pack.
	Image packs can be loaded from .wip files.

	@author WolfCoder
	@version 1
*/
public class ImagePack
{
	private FileInputStream fis;
	private DataInputStream dis;
	private int numFiles;
	private String name;
	private WolfcoderImage[] images;
	/**
		Creates a new Image Pack from loading a file.
		The file should be in .wip format.

		@param filename Name of the .wip file to load.
	*/
	public ImagePack(String filename)
	{
		// Start
		openWIP(filename);
		// Check file tag
		if(!checkTag())
		{
			System.out.println(filename+" is not a valid WIP2 file.");
			closeWIP(filename);
			return;
		}
		// Begin the loading process
		try
		{
			// Read the number of files
			numFiles = readInteger();
			// Read packname
			name = readString();
			// Allocate images
			images = new WolfcoderImage[numFiles];
			// Load each image
			for(int i = 0;i < numFiles;i++)
			{
				// Read required data
				String newImageName = readString();
				int numFrames = readInteger();
				int frameWidth = readInteger();
				int frameHeight = readInteger();
				// Check for positive
				if(frameWidth < 0)
					frameWidth = -frameWidth;
				if(frameHeight < 0)
					frameHeight = -frameHeight;
				int[] tempData = new int[frameWidth*frameHeight*4];
				// Create
				images[i] = new WolfcoderImage(newImageName,frameWidth,frameHeight,numFrames);
				// Write
				for(int j = 0;j < numFrames;j++)
				{
					for(int k = 0;k < frameWidth*frameHeight*4;k++)
						tempData[k] = dis.readUnsignedByte();
					images[i].write(tempData,j);
				}
			}
		}
		catch(Exception e)
		{
			System.out.println("Loading of "+filename+" has failed.");
			closeWIP(filename);
			return;
		}
		// Finsihed
		closeWIP(filename);
	}
	/**
		Reads a string in WIP2 format (ASCII).

		@return The constructed string from reading the next zero-terminated ASCII string in the file.
	*/
	private String readString()
	{
		char[] microBuffer = {(char)0x00,(char)0x00};
		String tempStr = new String();
		// Note: This method is very odd and possibly ineffecient, however, strings can only be concatenated using arrays.
		// Begin reading bytes into chars
		try
		{
			while(true)
			{
				// Read
				microBuffer[0] = (char)dis.readByte();
				// If zero, end.
				if(microBuffer[0] == '\0')
				{
					microBuffer[1] = (char)0x00;
					tempStr += new String(microBuffer);
					return tempStr.trim();
				}
				// Double on more chars
				microBuffer[1] = (char)dis.readByte();
				// Even zero terminate
				if(microBuffer[1] == '\0')
				{
					tempStr += new String(microBuffer);
					return tempStr.trim();
				}
				// Concat since not ending
				tempStr += new String(microBuffer);
			}
		}
		catch(Exception e)
		{
			System.out.println("Error reading WIP2 string using ASCII.");
		}
		// Error happens here
		return null;
	}
	/**
		Reads an integer in WIP2 format.
	
		@return The next integer in the file.
	*/
	private int readInteger()
	{
		int temp = 0;
		// Attempt to construct integer
		try
		{
			temp += dis.readByte();
			temp += dis.readByte()*256;
			temp += dis.readByte()*256*256;
			temp += dis.readByte()*256*256*256;
		}
		catch(Exception e)
		{
			System.out.println("Loading of a WIP2 integer failed.");
			return temp;
		}
		return temp;
	}
	/**
		Opens the WIP file for in-progress loading.

		@param filename Name of file to open.
	*/
	private void openWIP(String filename)
	{
		// Open
		File file = new File(filename);
		try
		{
			fis = new FileInputStream(file);
			dis = new DataInputStream(fis);
		}
		catch(Exception e)
		{
			System.out.println("Could not open image pack "+filename+".");
		}
	}
	/**
		Closes the in-progess loading of the file.

		@param filename Name of file to close.
	*/
	private void closeWIP(String filename)
	{
		// Close
		try
		{
			dis.close();
			fis.close();
		}
		catch(Exception e)
		{
			System.out.println("Could not close image pack "+filename+".");
		}
	}
	/**
		Checks the in-progess loading of the file for it's WIP2 tag.
	
		@return Whether or not this is a valid WIP file.
	*/
	private boolean checkTag()
	{
		try
		{
			// 4-char check
			if(dis.readByte() != 'W')
				return false;
			if(dis.readByte() != 'I')
				return false;
			if(dis.readByte() != 'P')
				return false;
			if(dis.readByte() != '2')
				return false;
		}
		catch(Exception e)
		{
			System.out.println("Could not read file tag.");
			return false;
		}
		// Passed checks
		return true;
	}
	/**
		Returns the array of WolfcoderImages in this pack.

		@return The array of WolfCoder Images.
	*/
	public WolfcoderImage[] getImages()
	{
		return images;
	}
	/**
		Returns a single WolfCoder Image matching the name given.
		Returns null if the name doesn't exist in this pack.

		@param name Name of the WolfCoder Image requested.
		@return The WolfCoder Image requested.
	*/
	public WolfcoderImage getImage(String name)
	{
		for(int i = 0;i < numFiles;i++)
		{
			// Find? return
			if(images[i].getName().equals(name))
				return images[i];
		}
		// Not found
		return null;
	}
	/**
		Returns the string representation of this ImagePack.

		@return The string representation of this ImagePack.
	*/
	public String toString()
	{
		return new String("WIP2: "+name+" ("+numFiles+" images)");
	}
}
