package nftile;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import nf.DataBlock;
import nfimage.BlockImage;
import nfimage.PaletteEntry;

public class Tileset extends DataBlock
{
	// Sizes
	public static final int COLORS = 12*16;
	public static final int PIXELS = 256*256;
	public static final int WIDTH = 256;
	public static final int HEIGHT = 256;
	public static final int TILES = 16*16;
	public static final int TILE_LENGTH = 16;
	// Types
	public static final int EMPTY = 0;
	public static final int NORMAL = 1;
	// Palette
	PaletteEntry[] pal = new PaletteEntry[COLORS];
	// Image data
	int[] pixel = new int[PIXELS];
	// Color map
	int[] color = new int[TILES];
	// Pass map
	int[] pass = new int[TILES];
	// Height map
	int[] level = new int[TILES];
	// Type map
	int[] type = new int[TILES];
	// Buffered
	BufferedImage preview;
	public Tileset()
	{
		update();
	}
	public BufferedImage getFragment(int i)
	{
		int x = i%32;
		int y = i/32;
		return preview.getSubimage(x*8,y*8,8,8);
	}
	// Redraw
	public void update()
	{
		preview = new BufferedImage(256,256,BufferedImage.TYPE_INT_ARGB);
		Graphics g = preview.createGraphics();
		// All tiles
		for(int ty = 0;ty < 16;ty++)
		{
			for(int tx = 0;tx < 16;tx++)
			{
				// Get palette
				int pali = color[tx+ty*16]*16;
				// Then, draw it
				for(int y = 0;y < 16;y++)
				{
					for(int x = 0;x < 16;x++)
					{
						int ix = tx*16+x+(ty*16+y)*256;
						int pix = pixel[ix];
						if(pal[pali+pix] != null && type[tx+ty*16] != EMPTY)
							g.setColor(pal[pali+pix].makeColor());
						else
							g.setColor(Color.BLACK);
						if(pix != 0)
							g.fillRect(x+tx*16,y+ty*16,1,1);
					}
				}
			}
		}
	}
	// Wipe out
	public boolean wipe(int tx,int ty)
	{
		// Empty?
		if(type[tx+ty*16] == EMPTY)
			return false;
		// Delete the palette entry
		int pali = color[tx+ty*16]*16;
		System.out.println("Removing palette entry "+pali);
		for(int i = 0;i < 16;i++)
			pal[pali+i] = null;
		// Delete color and type mappings
		for(int i = 0;i < TILES;i++)
		{
			// Erase it all if a match
			if(color[i] == pali/16)
			{
				System.out.print("T"+i+" ");
				color[i] = 0;
				type[i] = EMPTY;
				pass[i] = 0;
			}
		}
		System.out.println("deleted");
		// Done
		update();
		return true;
	}
	public BufferedImage getImage()
	{
		return preview;
	}
	// Has any more palettes left?
	public boolean hasNextPalette()
	{
		// Full
		if(getNextPalette() == -1)
			return false;
		// Has room
		return true;
	}
	// Get the next palette
	public int getNextPalette()
	{
		for(int i = 0;i < 12;i++)
		{
			// See if empty
			if(paletteIsEmpty(i))
				return i;
		}
		// Full
		return -1;
	}
	// Is the palette rank empty?
	private boolean paletteIsEmpty(int rank)
	{
		for(int i = 0;i < 16;i++)
		{
			if(pal[rank*16+i] != null)
			{
				if(pal[rank*16+i].r != 0 || pal[rank*16+i].g != 0 || pal[rank*16+i].b != 0)
				{
					return false;
				}
			}
		}
		return true;
	}
	// Open
	public void open(String filename) throws IOException
	{
		super.open(filename);
		update();
	}
	public int getType(int tx,int ty)
	{
		return type[tx+ty*16];
	}
	public int getPass(int tx,int ty)
	{
		return pass[tx+ty*16];
	}
	public void setPass(int tx,int ty)
	{
		// Toggle here
		if(pass[tx+ty*16] == 0)
			pass[tx+ty*16] = 1;
		else
			pass[tx+ty*16] = 0;
	}
	public void setPass(int tx,int ty,int c)
	{
		// Set here
		pass[tx+ty*16] = c;
	}
	public int getLevel(int tx,int ty)
	{
		return level[tx+ty*16];
	}
	public void setLevel(int tx,int ty)
	{
		// Toggle here
		if(level[tx+ty*16] == 0)
			level[tx+ty*16] = 1;
		else
			level[tx+ty*16] = 0;
	}
	public void setLevel(int tx,int ty,int c)
	{
		// Set here
		level[tx+ty*16] = c;
	}
	// Overlapping?
	private boolean overlapPlace(int tx,int ty,int w,int h)
	{
		for(int y = 0;y < h;y++)
		{
			for(int x = 0;x < w;x++)
			{
				int ix = tx+x+(ty+y)*16;
				// Overlapping?
				if(type[ix] != EMPTY)
					return true;
			}
		}
		// No overlap
		return false;
	}
	// Place the block image inside tileset
	public boolean place(BlockImage bi,int tx,int ty)
	{
		// Cancel?
		if(overlapPlace(tx,ty,bi.getWidth()/16,bi.getHeight()/16))
		{
			System.out.println("Overlapping tiles detected.");
			return false;
		}
		// Find the next open pal
		int pali = getNextPalette();
		// Write it
		System.out.println("Adding palette set "+pali);
		for(int i = 0;i < 16;i++)
			pal[pali*16+i] = bi.getPaletteData(i);
		// Copy tile pixels
		int txx = tx*16;
		int tyy = ty*16;
		System.out.println("Copying "+bi.getWidth()+" by "+bi.getHeight()+" to ("+txx+","+tyy+")");
		for(int y = 0;y < bi.getHeight();y++)
			for(int x = 0;x < bi.getWidth();x++)
				pixel[x+tx*16+(y+ty*16)*256] = bi.getPixelData(x,y);
		// Modify color map and type map
		int cmw = bi.getWidth()/16;
		int cmh = bi.getHeight()/16;
		System.out.println("Applying a "+cmw+" by "+cmh+" color map");
		for(int y = 0;y < cmh;y++)
		{
			for(int x = 0;x < cmw;x++)
			{
				color[x+tx+(y+ty)*16] = pali;
				type[x+tx+(y+ty)*16] = NORMAL;
			}
		}
		// Update
		update();
		return true;
	}
	private void writeBlock(DataOutputStream dos,int x,int y) throws IOException
	{
		for(int yy = 0;yy < 8;yy++)
			for(int xx = 0;xx < 8;xx++)
				writeNibble(dos,pixel[x+xx+(y+yy)*256]);
	}
	private void readBlock(DataInputStream dis,int x,int y) throws IOException
	{
		for(int yy = 0;yy < 8;yy++)
			for(int xx = 0;xx < 8;xx++)
				pixel[x+xx+(y+yy)*256] = readNibble(dis);
	}
	private void writeTile(DataOutputStream dos,int x,int y) throws IOException
	{
		writeBlock(dos,x,y);
		writeBlock(dos,x+8,y);
		writeBlock(dos,x,y+8);
		writeBlock(dos,x+8,y+8);
	}
	private void readTile(DataInputStream dis,int x,int y) throws IOException
	{
		readBlock(dis,x,y);
		readBlock(dis,x+8,y);
		readBlock(dis,x,y+8);
		readBlock(dis,x+8,y+8);
	}
	@Override
	public void write(DataOutputStream dos) throws IOException
	{
		// Write the palette
		for(int i = 0;i < COLORS;i++)
			PaletteEntry.write(dos,pal[i]);
		// Write the image
		for(int by = 0;by < TILE_LENGTH*2;by++)
			for(int bx = 0;bx < TILE_LENGTH*2;bx++)
				writeBlock(dos,bx*8,by*8);
		// Write the color map
		for(int i = 0;i < TILES;i++)
			writeInt(dos,color[i]);
		// Write the pass map
		for(int i = 0;i < TILES;i++)
			writeInt(dos,pass[i]);
		// Write the type map
		for(int i = 0;i < TILES;i++)
			writeInt(dos,type[i]);
		// Write the level map
		for(int i = 0;i < TILES;i++)
			writeInt(dos,level[i]);
	}
	@Override
	public void read(DataInputStream dis) throws IOException
	{
		// Read the palette
		for(int i = 0;i < COLORS;i++)
			pal[i] = PaletteEntry.read(dis);
		// Read in the image
		for(int by = 0;by < TILE_LENGTH*2;by++)
			for(int bx = 0;bx < TILE_LENGTH*2;bx++)
				readBlock(dis,bx*8,by*8);
		// Read the color map
		for(int i = 0;i < TILES;i++)
			color[i] = readInt(dis);
		// Read the pass map
		for(int i = 0;i < TILES;i++)
			pass[i] = readInt(dis);
		// Read the type map
		for(int i = 0;i < TILES;i++)
			type[i] = readInt(dis);
		// Read the level map
		for(int i = 0;i < TILES;i++)
			level[i] = readInt(dis);
	}
}