package nftile;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.AffineTransform;

import javax.swing.JPanel;

import nf.ErrorBox;
import nf.NovaDraw;
import nfimage.BlockImage;

public class TilePanel extends JPanel implements MouseListener,MouseMotionListener
{
	public static final int IMPORT_EDIT = 0;
	public static final int PASS_EDIT = 1;
	public static final int DELETE_EDIT = 2;
	public static final int LEVEL_EDIT = 3;
	private static final long serialVersionUID = 1L;
	private TileWindow tileParent;
	private Tileset tileset;
	private BlockImage clip;
	private int editMode;
	private int curX,curY;
	private int lastPass;
	private int lastLevel;
	public TilePanel(TileWindow tileParent)
	{
		// Set properties
		this.tileParent = tileParent;
		setBackground(Color.BLACK);
		setPreferredSize(new Dimension(256,256));
		addMouseListener(this);
		addMouseMotionListener(this);
	}
	public Tileset getFocus()
	{
		return tileset;
	}
	public void setEditMode(int em)
	{
		editMode = em;
		repaint();
	}
	public void focus(Tileset t)
	{
		// Reference and update
		tileset = t;
		repaint();
	}
	public void setClip(BlockImage bi)
	{
		clip = bi;
	}
	// Draw empty tile symbol
	private void drawEmptyTile(Graphics2D g,int tx,int ty)
	{
		tx *= 16;
		ty *= 16;
		g.setColor(new Color(32,32,32));
		g.drawLine(tx+2,ty+2,tx+16-4,ty+16-4);
		g.drawLine(tx+16-4,ty+2,tx+2,ty+16-4);
	}
	@Override
	public void paintComponent(Graphics g)
	{
		// Refresh by drawing the focused tileset
		super.paintComponent(g);
		if(tileset == null)
			return; // Do nothing on null
		Graphics2D g2d = (Graphics2D)g;
		int cx = curX*16;
		int cy = curY*16;
		// Draw the basic tileset
		g2d.drawRenderedImage(tileset.getImage(),AffineTransform.getTranslateInstance(0,0));
		// Draw all the types
		for(int y = 0;y < 16;y++)
		{
			for(int x = 0;x < 16;x++)
			{
				if(tileset.getType(x,y) == Tileset.EMPTY)
					drawEmptyTile(g2d,x,y);
			}
		}
		// Draw pass and solid?
		if(editMode == PASS_EDIT)
		{
			for(int y = 0;y < 16;y++)
			{
				for(int x = 0;x < 16;x++)
				{
					g.setColor(new Color(200,200,200));
					if(tileset.getType(x,y) != Tileset.EMPTY)
					{
						if(tileset.getPass(x,y) == 0)
							NovaDraw.drawOutlineString(g,"X",x*16+6,y*16+12);
						else
							NovaDraw.drawOutlineString(g,"O",x*16+6,y*16+12);
					}
				}
			}
		}
		// Draw upper and lower?
		if(editMode == LEVEL_EDIT)
		{
			for(int y = 0;y < 16;y++)
			{
				for(int x = 0;x < 16;x++)
				{
					g.setColor(new Color(200,200,200));
					if(tileset.getType(x,y) != Tileset.EMPTY)
					{
						if(tileset.getLevel(x,y) == 0)
							NovaDraw.drawOutlineString(g,".",x*16+6,y*16+12);
						else
							NovaDraw.drawOutlineString(g,"*",x*16+6,y*16+12);
					}
				}
			}
		}
		// Draw import logo if importing
		if(editMode == IMPORT_EDIT)
		{
			// Has a clip?
			if(clip != null)
			{
				// Draw top left
				g2d.setColor(Color.WHITE);
				g2d.drawLine(cx,cy,cx+16,cy);
				g2d.drawLine(cx,cy,cx,cy+16);
				g2d.drawString("I",curX*16+8,curY*16+14);
				// Draw bottom right
				int bx = cx+clip.getWidth();
				int by = cy+clip.getHeight();
				g2d.drawLine(bx,by,bx-16,by);
				g2d.drawLine(bx,by,bx,by-16);
			}
			else
			{
				// Draw import ghost
				g2d.setColor(Color.GRAY);
				g2d.drawLine(cx,cy,cx+8,cy);
				g2d.drawLine(cx,cy,cx,cy+8);
			}
		}
		if(editMode == DELETE_EDIT)
		{
			// Draw top left
			g2d.setColor(Color.RED);
			g2d.drawLine(cx,cy,cx+16,cy);
			g2d.drawLine(cx,cy,cx,cy+16);
			g2d.drawString("X",curX*16+8,curY*16+14);
		}
	}
	private void handleImport()
	{
		// Don't have a clip
		if(clip == null)
		{
			ErrorBox.show("Nothing imported yet.");
			return;
		}
		// Clip out of bounds
		if(clip.getWidth()/16+curX > 16 || clip.getHeight()/16+curY > 16)
		{
			ErrorBox.show("It won't fit there.");
			return;
		}
		// Empty clip into map
		if(tileset.place(clip,curX,curY))
		{
			clip = null;
			tileParent.setStatus("Tiles placed.");
		}
		else
		{
			ErrorBox.show("You can't place that on top of existing tiles.");
			return;
		}
	}
	@Override
	public void mouseClicked(MouseEvent e)
	{
		// No tileset
		if(tileset == null)
		{
			ErrorBox.show("No tileset loaded yet.");
			return;
		}
		// Delete mode
		if(editMode == DELETE_EDIT)
		{
			// Kill!
			if(!tileset.wipe(curX,curY))
				ErrorBox.show("Nothing to delete.");
			else
				tileParent.setStatus("Tiles deleted.");
		}
		repaint();
	}
	@Override
	public void mouseMoved(MouseEvent e)
	{
		// Get cursor
		curX = e.getX()/16;
		curY = e.getY()/16;
		repaint();
	}
	@Override
	public void mouseDragged(MouseEvent e)
	{
		// Get cursor
		curX = e.getX()/16;
		curY = e.getY()/16;
		if(editMode == PASS_EDIT)
		{
			// Just toggle
			tileset.setPass(curX,curY,lastPass);
		}
		if(editMode == LEVEL_EDIT)
		{
			// Just toggle
			tileset.setLevel(curX,curY,lastLevel);
		}
		repaint();
	}
	@Override
	public void mousePressed(MouseEvent e)
	{
		// Import mode
		if(editMode == IMPORT_EDIT)
			handleImport();
		if(editMode == PASS_EDIT)
		{
			// Just toggle
			tileset.setPass(curX,curY);
			lastPass = tileset.getPass(curX,curY);
		}
		if(editMode == LEVEL_EDIT)
		{
			// Just toggle
			tileset.setLevel(curX,curY);
			lastLevel = tileset.getLevel(curX,curY);
		}
	}
	@Override
	public void mouseReleased(MouseEvent e){}
	@Override
	public void mouseEntered(MouseEvent e){}
	@Override
	public void mouseExited(MouseEvent e){}
}