package nfmap;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import nf.DataBlock;
import nftile.Tileset;

public class TileMap extends DataBlock
{
	// Field numbers
	public static final int NUM_LAYERS = 2;
	// Layer codes
	public static final int LOWER_LAYER = 0;
	public static final int UPPER_LAYER = 1;
	// Size of the map
	private int width,height;
	// Every 8x8 fragment of the map
	private int[][] frag;
	// Tileset
	private Tileset tileset;
	// Previews
	private BufferedImage prev;
	private Graphics2D prevg;
	private boolean shadeUpper;
	public TileMap(int w,int h)
	{
		// Set dimension
		width = w;
		height = h;
		// Allocate
		frag = new int[NUM_LAYERS][width*height*4];
		prev = new BufferedImage(width*16,height*16,BufferedImage.TYPE_INT_ARGB);
		prevg = prev.createGraphics();
	}
	public TileMap()
	{
		// Default tilemap
		this(20,15);
	}
	public void setUpperShade(boolean bl)
	{
		// Set shadows
		shadeUpper = bl;
		updateAll();
	}
	// Sets a tile in layer LY at X and Y to value T
	public void setTile(int ly,int x,int y,int t)
	{
		// Out of bounds?
		if(x < 0 || x >= width || y < 0 || y >= height)
			return;
		// Double x and y
		x *= 2;
		y *= 2;
		// Set fragments
		frag[ly][x+y*width*2] = t;
		x += 1;
		t += 1;
		frag[ly][x+y*width*2] = t;
		t -= 1;
		x -= 1;
		t += 32;
		y += 1;
		frag[ly][x+y*width*2] = t;
		t += 1;
		x += 1;
		frag[ly][x+y*width*2] = t;
		// Update
		x /= 2;
		y /= 2;
		update(x,y);
	}
	// Return
	public int getFragment(int x,int y,int ly)
	{
		if(x < 0 || x >= width*2 || y < 0 || y >= height*2)
			return 0;
		// Return the fragment value
		return frag[ly][x+y*width*2];
	}
	// Draw this map on the graphic context
	public void draw(Graphics2D g)
	{
		g.drawRenderedImage(prev,AffineTransform.getTranslateInstance(0,0));
	}
	// Update only a single tile of the map
	public void update(int x,int y)
	{
		// Double it
		x *= 2;
		y *= 2;
		// Just want to paint it all black
		prevg.setColor(Color.BLACK);
		prevg.fillRect(x*8,y*8,16,16);
		// Draw it
		for(int i = 0;i < NUM_LAYERS;i++)
		{
			// Shade for the upper layer?
			if(shadeUpper)
			{
				// If above?
				if(i > 0)
				{
					// Shading rectangle
					prevg.setColor(new Color(0,0,0,128));
					prevg.fillRect(x*8,y*8,16,16);
				}
			}
			prevg.drawRenderedImage(tileset.getFragment(frag[i][x+y*width*2]),AffineTransform.getTranslateInstance(x*8,y*8));
			prevg.drawRenderedImage(tileset.getFragment(frag[i][x+1+y*width*2]),AffineTransform.getTranslateInstance(x*8+8,y*8));
			prevg.drawRenderedImage(tileset.getFragment(frag[i][x+(y+1)*width*2]),AffineTransform.getTranslateInstance(x*8,y*8+8));
			prevg.drawRenderedImage(tileset.getFragment(frag[i][(x+1)+(y+1)*width*2]),AffineTransform.getTranslateInstance(x*8+8,y*8+8));
		}
	}
	// Update entire map
	public void updateAll()
	{
		if(tileset == null)
			return;
		for(int y = 0;y < height;y++)
			for(int x = 0;x < width;x++)
				update(x,y);
	}
	public void setTileset(Tileset ts)
	{
		System.out.println("Now using "+ts+" as a tileset.");
		tileset = ts;
		// Update everything
		updateAll();
	}
	public int getWidth()
	{
		return width;
	}
	public int getHeight()
	{
		return height;
	}
	@Override
	public void write(DataOutputStream dos) throws IOException
	{
		// Write the width and height of the map
		writeInt(dos,width);
		writeInt(dos,height);
		// Write the fragments
		for(int i = 0;i < NUM_LAYERS;i++)
			for(int j = 0;j < width*height*4;j++)
				writeInt(dos,frag[i][j]);
	}
	@Override
	public void read(DataInputStream dis) throws IOException
	{
		// Read the width and height of the map
		width = readInt(dis);
		height = readInt(dis);
		// Read the fragments
		for(int i = 0;i < NUM_LAYERS;i++)
			for(int j = 0;j < width*height*4;j++)
				frag[i][j] = readInt(dis);
		// Reset
		updateAll();
	}
}