package nfimage;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;

import nf.DataBlock;
import nf.ErrorBox;

/**
 * A block image is the particular kind of format used in the .nfi format.
 * This method is typically designed for systems such as the GBA.
 * The width and height of the image must be a multiple of 8.
 * @author Sayuri
 */
public class BlockImage extends DataBlock
{
	public static final int IMAGE_16 = 0;
	public static final int IMAGE_256 = 1;
	public static final int IMAGE_COLOR = 2;
	private int type;
	private int width,height;
	private int frameWidth,frameHeight;
	private PaletteEntry[] pal;
	private int[] pixels;
	private ColorTriple[] pixels24;
	private BufferedImage preview;
	private BufferedImage raw;
	private ArrayList<ImageListener> ail = new ArrayList<ImageListener>();
	public BufferedImage getSubImage(int x,int y,int w,int h)
	{
		return raw.getSubimage(x,y,w,h);
	}
	/**
	 * Generates the preview buffered image for speed reasons.
	 */
	private void update()
	{
		// Is there even data?
		if(width == 0 || height == 0)
			return;
		// Prepare
		preview = new BufferedImage(width,height,BufferedImage.TYPE_INT_RGB);
		Graphics g = preview.createGraphics();
		// Draw
		for(int y = 0;y < height;y++)
		{
			for(int x = 0;x < width;x++)
			{
				g.setColor(getPixel(x,y));
				g.fillRect(x,y,1,1);
			}
		}
		// Draw grids
		if(frameWidth != 0 && frameHeight != 0)
		{
			for(int y = 0;y < height;y += frameHeight*8)
			{
				for(int x = 0;x < width;x += frameWidth*8)
				{
					g.setColor(new Color(255,255,255,192));
					g.drawLine(x,y,x+frameWidth*8,y);
					g.drawLine(x,y,x,y+frameHeight*8);
				}
			}
		}
		// Notify
		for(int i = 0;i < ail.size();i++)
			ail.get(i).imageChanged(preview);
	}
	public BufferedImage makeRaw()
	{
		// Is there even data?
		if(width == 0 || height == 0)
			return null;
		// Prepare
		raw = new BufferedImage(width,height,BufferedImage.TYPE_INT_RGB);
		Graphics g = raw.createGraphics();
		// Draw
		for(int y = 0;y < height;y++)
		{
			for(int x = 0;x < width;x++)
			{
				if(!isTransparent(x,y))
					g.setColor(getPixel(x,y));
				g.fillRect(x,y,1,1);
			}
		}
		// Return it
		return raw;
	}
	/**
	 * Reads an integer backwards (reversed endian).
	 * @param dis The source stream
	 * @return The read integer decoded backwards
	 * @throws IOException
	 */
	private int readBackwardsInt(DataInputStream dis) throws IOException
	{
		// Build
		int accu = (dis.readByte()&0xFF);
		accu += (dis.readByte()&0xFF)<<8;
		accu += (dis.readByte()&0xFF)<<16;
		accu += (dis.readByte()&0xFF)<<24;
		// Return
		return accu;
	}
	/**
	 * Reads a short backwards (reversed endian).
	 * @param dis The source stream
	 * @return The read short decoded backwards
	 * @throws IOException
	 */
	private int readBackwardsShort(DataInputStream dis) throws IOException
	{
		// Build
		int accu = (dis.readByte()&0xFF);
		accu += (dis.readByte()&0xFF)<<8;
		// Return
		return accu;
	}
	/**
	 * Holds a single color triple.
	 * @author Sayuri
	 */
	private class ColorTriple
	{
		private int r,g,b;
		/**
		 * Reads the color triple from an input stream.
		 * @param dis Source stream
		 * @throws IOException
		 */
		public ColorTriple(DataInputStream dis) throws IOException
		{
			// Read it
			b = dis.readUnsignedByte();
			g = dis.readUnsignedByte();
			r = dis.readUnsignedByte();
		}
		/**
		 * Returns the color of this pixel triple.
		 * @return The color matching this triple
		 */
		public Color makeColor()
		{
			// Make
			return new Color(r,g,b);
		}
	}
	/**
	 * Holds the .bmp file header data.
	 * @author Sayuri
	 */
	private class BitmapFileHeader
	{
		public char[] tag = new char[2];
		public int size;
		//public int r1,r2;
		public int offbits;
		/**
		 * Creates a new bitmap file header read in from a stream.
		 * @param dis The source stream
		 * @throws IOException
		 */
		public BitmapFileHeader(DataInputStream dis) throws IOException
		{
			// Read the file tag
			tag[0] = (char)dis.readByte();
			tag[1] = (char)dis.readByte();
			// Read the file size
			size = readBackwardsInt(dis);
			readBackwardsShort(dis);
			readBackwardsShort(dis);
			// Read the bit offset
			offbits = readBackwardsInt(dis);
		}
		/**
		 * Returns whether or not this is a valid .bmp file
		 * @return True if "BM" was found
		 */
		public boolean valid()
		{
			// Does it say BM?
			if(tag[0] == 'B' && tag[1] == 'M')
				return true;
			// Invalid file
			return false;
		}
	}
	/**
	 * Holds the .bmp info header data.
	 * @author Sayuri
	 */
	private class BitmapInfoHeader
	{
		//public int structSize;
		public int width;
		public int height;
		//public int planes;
		public int bpp;
		//public int compression;
		//public int sizeImage;
		//public int xdpi,ydpi;
		//public int colorUsed;
		//public int colorImportant;
		/**
		 * Creates a new .bmp info header from the input stream.
		 * @param dis Source stream
		 * @throws IOException
		 */
		public BitmapInfoHeader(DataInputStream dis) throws IOException
		{
			// Read in
			readBackwardsInt(dis);
			width = readBackwardsInt(dis);
			height = readBackwardsInt(dis);
			readBackwardsShort(dis);
			bpp = readBackwardsShort(dis);
			readBackwardsInt(dis);
			readBackwardsInt(dis);
			readBackwardsInt(dis);
			readBackwardsInt(dis);
			readBackwardsInt(dis);
			readBackwardsInt(dis);
		}
	}
	/**
	 * Creates a new block image for loading purposes.
	 */
	public BlockImage()
	{
		// Blank
		this(16,16,0);
		// Update
		update();
	}
	public BlockImage(int w,int h,int t)
	{
		// Check for error
		if(w%8 != 0 || h%8 != 0)
			throw new IllegalArgumentException();
		// Set type
		type = t;
		// Allocate palette
		if(t == IMAGE_16)
			pal = new PaletteEntry[16];
		if(t == IMAGE_256)
			pal = new PaletteEntry[256];
		// Fill with blacks
		for(int i = 0;i < pal.length;i++)
			pal[i] = new PaletteEntry();
		// Allocate pixels
		if(t == IMAGE_COLOR)
			pixels24 = new ColorTriple[w*h];
		else
			pixels = new int[w*h];
		// W and H
		width = w;
		height = h;
	}
	/**
	 * Creates a new block image from the given .bmp file.
	 * @param filename The filename to load
	 * @throws IOException
	 */
	public BlockImage(String filename) throws IOException
	{
		// Open channel
		FileInputStream fis = new FileInputStream(filename);
		DataInputStream dis = new DataInputStream(fis);
		// Read data
		BitmapFileHeader bfh = new BitmapFileHeader(dis);
		if(!bfh.valid())
			throw new IOException();
		BitmapInfoHeader bih = new BitmapInfoHeader(dis);
		// Report the size of the bitmap
		System.out.println(filename+" loaded as a "+bfh.size+" byte .bmp file.");
		// Not yet supported full color
		if(bih.bpp != 8)
		{
			ErrorBox.show("24-bit color .BMP files are not supported yet.");
			throw new IOException();
		}
		// Width and height must be multiples of 8
		if(bih.width%8 != 0 || bih.height%8 != 0)
		{
			ErrorBox.show("Width and height must be multiples of 8.");
			throw new IOException();
		}
		width = bih.width;
		height = bih.height;
		System.out.println(width+" by "+height+" pixels in size.");
		// Calculate the offsets
		int palSize = bfh.offbits-(14+40);
		if(palSize > 16*4)
		{
			type = IMAGE_256;
		}
		palSize /= 4;
		System.out.println("Palette is "+palSize+" colors long.");
		// Load the palette
		pal = new PaletteEntry[256];
		for(int i = 0;i < palSize;i++)
			pal[i] = new PaletteEntry(dis);
		// Check for garbages
		if(palSize == 16)
		{
			// Reduce
			if(pal[15].r == 0 && pal[15].g == 0 && pal[15].b == 0)
			{
				System.out.println("Image is one-less than 16 colors.");
				palSize--;
			}
		}
		// Reset back to 16 color if the palette is overtruncated
		if(palSize > 16)
		{
			// Set for 16 for the time being
			type = IMAGE_16;
			// Check for any non-zero pal entires
			for(int i = 16;i < palSize;i++)
			{
				if(pal[i].r != 0 && pal[i].g != 0 && pal[i].b != 0)
				{
					System.out.println("Image has a short palette, but a valid palette.");
					type = IMAGE_256;
				}
			}
		}
		// Finally announce import format
		if(type == IMAGE_16)
			System.out.println("Image will be in 16 colors.");
		if(type == IMAGE_256)
			System.out.println("Image will be in 256 colors.");
		// RGB?
		if(bih.bpp == 24)
		{
			// Load the 24-bit pixels
			type = IMAGE_COLOR;
			pixels24 = new ColorTriple[bih.width*bih.height];
			System.out.println("Image is true-color with "+pixels24.length+" pixels.");
			for(int y = 0;y < height;y++)
				for(int x = 0;x < width;x++)
					pixels24[x+(height-y-1)*width] = new ColorTriple(dis);
		}
		else
		{
			// Load the pixels
			pixels = new int[bih.width*bih.height];
			System.out.println("Image is "+pixels.length+" pixels big.");
			for(int y = 0;y < height;y++)
				for(int x = 0;x < width;x++)
					pixels[x+(height-y-1)*width] = dis.readUnsignedByte();
		}
		// Expand palette for no-transparency if it is less than 16 colors
		if(palSize < 16 && palSize != 0)
		{
			System.out.println("No transparent default included.");
			// Shift all right one
			for(int i = 15;i > 0;i--)
				pal[i] = pal[i-1];
			// Replace first with trans
			pal[0] = new PaletteEntry();
			// Add 1 to all pixels
			for(int i = 0;i < pixels.length;i++)
				pixels[i] += 1;
		}
		// Close
		dis.close();
		fis.close();
		// Preview
		update();
	}
	/**
	 * Returns width of image.
	 * @return The width
	 */
	public int getWidth()
	{
		// Get
		return width;
	}
	/**
	 * Returns height of image.
	 * @return The height
	 */
	public int getHeight()
	{
		// Get
		return height;
	}
	/**
	 * Returns a color matching the color of a pixel at the location given.
	 * @param x The x location of pixel
	 * @param y The y location of pixel
	 * @return Color of that pixel
	 */
	public Color getPixel(int x,int y)
	{
		// Return index color
		if(pixels != null)
			return pal[pixels[x+y*width]].makeColor();
		// Return direct color
		return pixels24[x+y*width].makeColor();
	}
	/**
	 * Returns true if a given pixel is transparent.
	 * @param x The x location of pixel
	 * @param y The y location of pixel
	 * @return Whether or not the pixel is transparent
	 */
	public boolean isTransparent(int x,int y)
	{
		// Return index color
		if(pixels != null)
		{
			return (pixels[x+y*width] == 0);
		}
		// Return direct color
		return true; // Null pixels are transparent
	}
	/**
	 * Returns the preview image for drawing purposes.
	 * @return The BufferedImage representing this BlockImage
	 */
	public BufferedImage getImage()
	{
		// Get
		return preview;
	}
	/**
	 * Adds an image listener.
	 * @param il The image listener to add
	 */
	public void addImageListener(ImageListener il)
	{
		// Add
		ail.add(il);
	}
	/**
	 * Removes an image listener
	 * @param il The image listener to remove
	 */
	public void removeImageListener(ImageListener il)
	{
		// Remove
		ail.remove(ail.indexOf(il));
	}
	/**
	 * Change the frame values.
	 * @param fw New width of frame
	 * @param fh New height of frame
	 */
	public void setFrame(int fw,int fh)
	{
		frameWidth = fw;
		frameHeight = fh;
		update();
	}
	@Override
	public void open(String filename) throws IOException
	{
		// Open
		super.open(filename);
		// Update
		update();
	}
	public int getFrameWidth()
	{
		return frameWidth;
	}
	public int getFrameHeight()
	{
		return frameHeight;
	}
	private void readPixelBlock(DataInputStream dis,int x,int y) throws IOException
	{
		for(int by = 0;by < 8;by++)
		{
			for(int bx = 0;bx < 8;bx++)
			{
				int index = (bx+x)+(by+y)*width;
				// Write a pixel in either mode
				if(type == IMAGE_16)
					pixels[index] = readNibble(dis);
				else
					pixels[index] = dis.readUnsignedByte();
			}
		}
	}
	private void readFrameBlock(DataInputStream dis,int x,int y) throws IOException
	{
		for(int fy = 0;fy < frameHeight*8;fy += 8)
			for(int fx = 0;fx < frameWidth*8;fx += 8)
				readPixelBlock(dis,fx+x,fy+y);
	}
	/**
	 * Writes an 8x8 pixel block to the stream.
	 * @param dos Output stream
	 * @param x X position in image
	 * @param y Y position in image
	 * @throws IOException
	 */
	private void writePixelBlock(DataOutputStream dos,int x,int y) throws IOException
	{
		for(int by = 0;by < 8;by++)
		{
			for(int bx = 0;bx < 8;bx++)
			{
				int index = (bx+x)+(by+y)*width;
				// Write a pixel in either mode
				if(type == IMAGE_16)
					writeNibble(dos,(byte)pixels[index]);
				else
					dos.writeByte((byte)pixels[index]);
			}
		}
	}
	/**
	 * Writes an mxn frame block to the stream.
	 * @param dos Output stream
	 * @param x X position in image
	 * @param y Y position in image
	 * @throws IOException
	 */
	private void writeFrameBlock(DataOutputStream dos,int x,int y) throws IOException
	{
		for(int fy = 0;fy < frameHeight*8;fy += 8)
			for(int fx = 0;fx < frameWidth*8;fx += 8)
				writePixelBlock(dos,fx+x,fy+y);
	}
	public Color getTransparentColor()
	{
		// Invalid palette?
		if(pal == null)
			return Color.BLACK; // Use black as nullcolor
		// Make it and return
		return pal[0].makeColor();
	}
	public int getPixelData(int x,int y)
	{
		return pixels[x+y*width];
	}
	public PaletteEntry getPaletteData(int i)
	{
		return pal[i];
	}
	public void setTransparentColor(int x,int y)
	{
		// Out of bounds?
		if(x >= width || x < 0 || y >= height || y < 0)
		{
			ErrorBox.show("Please click inside the image to select transparent color.");
			return;
		}
		// Get the color index
		int indexed = pixels[x+y*width];
		// Remember the old color in first
		PaletteEntry ptemp = pal[0];
		// Replace old color
		pal[0] = pal[indexed];
		pal[indexed] = ptemp;
		// Replace pixels
		for(int i = 0;i < pixels.length;i++)
		{
			if(pixels[i] == 0)
				pixels[i] = indexed;
			else if(pixels[i] == indexed)
				pixels[i] = 0;
		}
		// Colors have been swapped, now refresh
		update();
	}
	// Returns the raw
	public BufferedImage getRaw()
	{
		return raw;
	}
	public void writeImage(DataOutputStream dos) throws IOException
	{
		// Output the image data itself
		if(pixels != null)
		{
			// Write them out
			if(frameHeight != 0)
			{
				for(int by = 0;by < height;by += frameHeight*8)
					for(int bx = 0;bx < width;bx += frameWidth*8)
						writeFrameBlock(dos,bx,by);
			}
			else
			{
				for(int by = 0;by < height;by += 8)
					for(int bx = 0;bx < width;bx += 8)
						writePixelBlock(dos,bx,by);
			}
		}
	}
	public void readImage(DataInputStream dis) throws IOException
	{
		// Alloc
		pixels = new int[width*height];
		// Read them in
		if(frameHeight != 0)
		{
			for(int by = 0;by < height;by += frameHeight*8)
				for(int bx = 0;bx < width;bx += frameWidth*8)
					readFrameBlock(dis,bx,by);
		}
		else
		{
			for(int by = 0;by < height;by += 8)
				for(int bx = 0;bx < width;bx += 8)
					readPixelBlock(dis,bx,by);
		}
	}
	public int getType()
	{
		return type;
	}
	public void writePaletteNum(DataOutputStream dos,int n) throws IOException
	{
		// Write colors
		for(int i = 0;i < n;i++)
			PaletteEntry.write(dos,pal[i]);
	}
	public void writePalette(DataOutputStream dos) throws IOException
	{
		if(type == IMAGE_16)
			writePaletteNum(dos,16);
		else
			writePaletteNum(dos,256);
	}
	public void readPaletteNum(DataInputStream dis,int n) throws IOException
	{
		// Report
		System.out.println("Reading "+n+" colors for palette.");
		pal = new PaletteEntry[n];
		// Read
		for(int i = 0;i < pal.length;i++)
			pal[i] = PaletteEntry.read(dis);
	}
	public void readPalette(DataInputStream dis) throws IOException
	{
		if(type == IMAGE_16)
			readPaletteNum(dis,16);
		else
			readPaletteNum(dis,256);
	}
	@Override
	public void write(DataOutputStream dos) throws IOException
	{
		// Write the type code
		writeInt(dos,type);
		System.out.println("Saved as image type "+type);
		// Write out the blocks in sizes
		writeInt(dos,width/8);
		writeInt(dos,height/8);
		// Write out the frame sizes
		writeInt(dos,frameWidth);
		writeInt(dos,frameHeight);
		// Write out the frame block size for each frame
		writeInt(dos,frameWidth*frameHeight);
		writePalette(dos);
		writeImage(dos);
	}
	@Override
	public void read(DataInputStream dis) throws IOException
	{
		// Read the type code
		type = readInt(dis);
		// Read the dimensions
		width = readInt(dis)*8;
		height = readInt(dis)*8;
		// Report this
		System.out.println("Reading NFI: "+type+" ("+width+","+height+")");
		// Read the frame sizes
		frameWidth = readInt(dis);
		frameHeight = readInt(dis);
		readInt(dis);
		// Report this
		System.out.println("Frame size: ("+frameWidth+","+frameHeight+")");
		// Read the frame total size but ignore it
		readPalette(dis);
		readImage(dis);
	}
}