package nf;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

/**
 * A data block is a block of NovaForce-platform data.
 * It can be written and read from streams.
 * @author Sayuri
 */
public abstract class DataBlock
{
	private byte buffer4;
	private boolean hasBuffer4;
	public void save(String filename) throws IOException
	{
		FileOutputStream fos = new FileOutputStream(filename);
		DataOutputStream dos = new DataOutputStream(fos);
		write(dos);
		dos.flush();
		dos.close();
		fos.close();
	}
	public void open(String filename) throws IOException
	{
		FileInputStream fis = new FileInputStream(filename);
		DataInputStream dis = new DataInputStream(fis);
		read(dis);
		dis.close();
		fis.close();
	}
	/**
	 * Writes all the data to the stream given.
	 * @param dos Output stream
	 * @throws IOException
	 */
	public abstract void write(DataOutputStream dos) throws IOException;
	/**
	 * Reads all the data from the stream given.
	 * @param dis Input stream
	 * @throws IOException
	 */
	public abstract void read(DataInputStream dis) throws IOException;
	/**
	 * Reads all the data from the stream given.
	 * If there is a problem, instead of failing, false is returned.
	 * If this operation succeeds, true is returned.
	 * @param dis Input stream
	 * @return Whether or not this operation worked
	 */
	public boolean load(DataInputStream dis)
	{
		// Attept
		try
		{
			// Read
			read(dis);
		}
		catch(IOException e)
		{
			// Fail
			return false;
		}
		// Pass
		return true;
	}
	/**
	 * Writes the integer as a strict 4-byte value.
	 * @param dos The output stream
	 * @param i The number to write
	 * @throws IOException
	 */
	protected void writeInt(DataOutputStream dos,int i) throws IOException
	{
		// Chop and write
		dos.writeByte((byte)(i));
		dos.writeByte((byte)(i>>>8));
		dos.writeByte((byte)(i>>>16));
		dos.writeByte((byte)(i>>>24));
	}
	/**
	 * Reads a strictly 4-byte integer.
	 * @param dis The input stream
	 * @return The read integer
	 * @throws IOException
	 */
	protected int readInt(DataInputStream dis) throws IOException
	{
		// Read bytes and build
		int accu = 0;
		accu += (dis.readByte()&0xFF);
		accu += (dis.readByte()&0xFF)<<8;
		accu += (dis.readByte()&0xFF)<<16;
		accu += (dis.readByte()&0xFF)<<24;
		// Return
		return accu;
	}
	/**
	 * Writes a 4-bit nibble.
	 * @param dos The output stream
	 * @param n The nibble to write
	 * @throws IOException
	 */
	protected void writeNibble(DataOutputStream dos,int n) throws IOException
	{
		// Finish writing
		if(hasBuffer4)
		{
			dos.writeByte(buffer4|(n<<4));
			hasBuffer4 = false;
			return;
		}
		// Load buffer
		buffer4 = (byte)n;
		hasBuffer4 = true;
	}
	/**
	 * Reads a 4-bit nibble.
	 * @param dis The input stream
	 * @return The nibble read
	 * @throws IOException
	 */
	protected int readNibble(DataInputStream dis) throws IOException
	{
		// Has buffer?
		if(hasBuffer4)
		{
			// Release
			hasBuffer4 = false;
			//return (byte)(buffer4&0x0F);
			return (buffer4>>>4)&0x0F;
		}
		// Doesn't have buffer
		buffer4 = dis.readByte();
		hasBuffer4 = true;
		//return (byte)((buffer4>>>4)+16);
		return (buffer4&0x0F);
	}
}