/*
	Image Loader
*/

/* Includes */
#include "gbaimg.h"
#include <stdio.h>
#include <math.h>
#include <malloc.h>

/* Bitmap file header */
typedef struct
{
	unsigned long file_size;
	unsigned short res1;
	unsigned short res2;
	unsigned long offset;
}bitmap_file_header;

/* Bitmap info header */
typedef struct
{
	unsigned long size_struct;
	long width;
	long height;
	unsigned short planes;
	unsigned short bpp;
	unsigned long compression;
	unsigned long image_size;
	long xdpi;
	long ydpi;
	unsigned long color_used;
	unsigned long color_important;
}bitmap_info_header;

/* Bitmap palette */
typedef struct
{
	unsigned char blue;
	unsigned char green;
	unsigned char red;
	unsigned char flags;
}bitmap_color;

/* Loads the target bitmap into memory */
int gbaimg_load(image *img,char *filename)
{
	bitmap_file_header bfh;
	bitmap_info_header bih;
	bitmap_color col[256];
	unsigned char tag[2];
	FILE *f;
	int i;
	int palsize;
	/* Open the file */
	fopen_s(&f,filename,"rb");
	if(!f)
	{
		fprintf(stderr,"Could not open %s.\n",filename);
		return 0;
	}
	/* Check the tag */
	fread(tag,1,2,f);
	if(tag[0] != 'B' || tag[1] != 'M')
	{
		fprintf(stderr,"%s is not a valid bitmap file.\n",filename);
		fclose(f);
		return 0;
	}
	/* Read the file header */
	fread(&bfh,sizeof(bfh),1,f);
	/* Read the info header */
	fread(&bih,sizeof(bih),1,f);
	/* Check the bpp */
	if(bih.bpp != 8 && bih.bpp != 24)
	{
		fprintf(stderr,"%s is not in 8-bit or 24-bit color.\n",filename);
		fclose(f);
		return 0;
	}
	/* Check the sizes */
	if(bih.width%8 != 0 || bih.height%8 != 0)
	{
		fprintf(stderr,"%s does not have width or height a multiple of 8.\n",filename);
		fclose(f);
		return 0;
	}
	/* Check compression */
	if(bih.compression != 0)
	{
		fprintf(stderr,"%s is not uncompressed.\n",filename);
		fclose(f);
		return 0;
	}
	/* Read colors? */
	if(bih.bpp == 8)
	{
		/* Calculate palsize */
		palsize = bfh.offset-sizeof(bfh)-2-bih.size_struct;
		palsize /= 4;
		/* Read */
		fread(col,sizeof(bitmap_color),palsize,f);
	}
	/* Set image data */
	img->w = abs(bih.width);
	img->h = abs(bih.height);
	if(bih.bpp == 8)
		img->model = IMAGE_INDEX_MODEL;
	else
		img->model = IMAGE_FULL_COLOR_MODEL;
	/* Allocate and read */
	if(img->model == IMAGE_INDEX_MODEL)
	{
		/* Malloc */
		img->palette = (color*)malloc(256*sizeof(color));
		img->data = (unsigned char*)malloc(img->w*img->h);
		/* Read*/
		fread(img->data,1,img->w*img->h,f);
		/* Set palette */
		for(i = 0;i < 256;i++)
		{
			img->palette[i].red = col[i].red;
			img->palette[i].green = col[i].green;
			img->palette[i].blue = col[i].blue;
		}
	}
	else
	{
		/* Malloc */
		img->cdata = (color*)malloc(img->w*img->h*sizeof(color));
		/* Read */
		fread(img->cdata,sizeof(color),img->w*img->h,f);
	}
	/* Done */
	fclose(f);
	return 1;
}

/* Free an image */
void gbaimg_free(image *img)
{
	/* Select model */
	if(img->model == IMAGE_INDEX_MODEL)
	{
		/* Free index data */
		free(img->palette);
		free(img->data);
	}
	else
	{
		/* Free color data */
		free(img->cdata);
	}
}

/* Returns color model name */
char *gbaimg_name(int model)
{
	/* Index */
	if(model == IMAGE_INDEX_MODEL)
		return "Index";
	/* Full */
	return "FullColor";
}
