/*
	Image Exporter
*/

/* Includes */
#include "gbaexp.h"
#include <stdio.h>

/* Returns the name */
char *gbaexp_name(int method)
{
	/* 16-color */
	if(method == METHOD_4)
		return "Index16";
	/* 256-color */
	if(method == METHOD_8)
		return "Index256";
	/* Full color */
	return "FullColor16";
}

/* Exports the image using 16-bit color */
void gbaexp_16(image *img,FILE *f)
{
	int x,y;
	color col;
	unsigned short pix;
	/* Now convert the picture data */
	for(y = img->h-1;y >= 0;y--)
	{
		for(x = 0;x < img->w;x++)
		{
			/* Get the color */
			col = img->cdata[x+y*img->w];
			/* Reduce the color */
			col.red /= 8;
			col.green /= 8;
			col.blue /= 8;
			/* Pack the color */
			pix = col.red+(col.green<<5)+(col.blue<<10);
			/* Write the color */
			fwrite(&pix,2,1,f);
		}
	}
}

/* Writes a block of pixels in 8-bit mode */
void gbaexp_8_block(image *img,int x,int y,FILE *f)
{
	int xx,yy;
	unsigned char b;
	/* Write them */
	for(yy = 0;yy < 8;yy++)
	{
		for(xx = 0;xx < 8;xx++)
		{
			/* Get pixel */
			b = img->data[x+xx+(img->h-(y+yy)-1)*img->w];
			/* Write pixel */
			fwrite(&b,1,1,f);
		}
	}
}

/* Exports the image using 8-bit color */
void gbaexp_8(image *img,FILE *f)
{
	int i;
	color col;
	unsigned short pix;
	int bx,by;
	/* First, we export the entire palette */
	for(i = 0;i < 256;i++)
	{
		/* Get the color */
		col = img->palette[i];
		/* Reduce the color */
		col.red /= 8;
		col.green /= 8;
		col.blue /= 8;
		/* Pack the color */
		pix = col.red+(col.green<<5)+(col.blue<<10);
		/* Write the color */
		fwrite(&pix,2,1,f);
	}
	/* Next, we export the 8-bit data in block format */
	for(by = 0;by < img->h;by += 8)
		for(bx = 0;bx < img->w;bx += 8)
			gbaexp_8_block(img,bx,by,f);
}

/* Writes a block of pixels in 8-bit mode */
void gbaexp_4_block(image *img,int x,int y,FILE *f)
{
	int xx,yy;
	unsigned char b,c;
	unsigned char packed;
	/* Write them */
	for(yy = 0;yy < 8;yy++)
	{
		for(xx = 0;xx < 8;xx += 2)
		{
			/* Get two pixels */
			b = img->data[x+xx+(img->h-(y+yy)-1)*img->w];
			c = img->data[x+xx+(img->h-(y+yy)-1)*img->w+1];
			/* Pack them */
			packed = b+(c<<4);
			/* Write pixel */
			fwrite(&packed,1,1,f);
		}
	}
}

/* Exports the image using 4-bit color */
void gbaexp_4(image *img,FILE *f)
{
	int i;
	color col;
	unsigned short pix;
	int bx,by;
	/* First, we export the entire palette */
	for(i = 0;i < 16;i++)
	{
		/* Get the color */
		col = img->palette[i];
		/* Reduce the color */
		col.red /= 8;
		col.green /= 8;
		col.blue /= 8;
		/* Pack the color */
		pix = col.red+(col.green<<5)+(col.blue<<10);
		/* Write the color */
		fwrite(&pix,2,1,f);
	}
	/* Next, we export the 4-bit data in block format */
	for(by = 0;by < img->h;by += 8)
		for(bx = 0;bx < img->w;bx += 8)
			gbaexp_4_block(img,bx,by,f);
}

/* Converts the image */
void gbaexp_conv(image *img,char *filename,int mode)
{
	FILE *f;
	unsigned short ucode;
	unsigned short cx,cy,cn;
	/* Open file */
	fopen_s(&f,filename,"wb");
	/* Convert code */
	ucode = (unsigned short)mode;
	fwrite(&ucode,2,1,f);
	/* Number of cells we have */
	cx = img->w/8;
	cy = img->h/8;
	cn = cx*cy;
	/* Write them */
	fwrite(&cx,2,1,f);
	fwrite(&cy,2,1,f);
	fwrite(&cn,2,1,f);
	/* Do it */
	if(mode == METHOD_16)
		gbaexp_16(img,f);
	if(mode == METHOD_8)
		gbaexp_8(img,f);
	if(mode == METHOD_4)
		gbaexp_4(img,f);
	/* Close */
	fclose(f);
}
