/*
	GameBoy Advance Bitmap Tool
	reads .bmp files and converts them to .gbp files

	written by Sayuri L. Kitsune (2010)
*/

/* Include */
#include "gbaimg.h"
#include "gbaexp.h"
#include "gbaconv.h"
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <memory.h>

/* Specified output configuration */
int option_cmode = -1; /* No change */
int trans_red = 0;
int trans_green = 0;
int trans_blue = 0;

/* Swaps the trans color with the first color of a palette */
void gbabmp_trans_color(image *img)
{
	int i;
	int repcolor;
	color col;
	color tcol;
	/* Zero set */
	repcolor = 0;
	/* Swap */
	for(i = 0;i < 256;i++)
	{
		/* Check */
		col = img->palette[i];
		if(col.red == trans_red && col.blue == trans_blue && col.green == trans_green)
		{
			tcol = img->palette[0];
			img->palette[0] = img->palette[i];
			img->palette[i] = tcol;
			repcolor = i;
		}
	}
	/* Replace indecies */
	for(i = 0;i < img->w*img->h;i++)
	{
		/* Swap colors if trans */
		if(img->data[i] == repcolor)
		{
			img->data[i] = 0;
		}
		else
		{
			if(img->data[i] == 0)
				img->data[i] = repcolor;
		}
	}
}

/* Checks an option define */
void gbabmp_option(char *op)
{
	/* Not an option */
	if(op[0] != '-')
		return;
	/* Check kind of option */
	if(op[1] == 'c')
	{
		/* Color target operation */
		option_cmode = atoi(&op[2]);
		/* Check validity */
		if(option_cmode != 4 && option_cmode != 8 && option_cmode != 16)
		{
			printf("ERROR: Invalid target color bit depth: %d.\n",option_cmode);
			option_cmode = -1;
		}
	}
	if(op[1] == 'r')
	{
		/* Transparent color red */
		trans_red = atoi(&op[2]);
	}
	if(op[1] == 'g')
	{
		/* Transparent color green */
		trans_green = atoi(&op[2]);
	}
	if(op[1] == 'b')
	{
		/* Transparent color blue */
		trans_blue = atoi(&op[2]);
	}
}

/* Converts a filename */
char *gbabmp_outfile(char *infile)
{
	char *c;
	int len;
	/* Duplicate */
	len = (int)strlen(infile)+1;
	c = (char*)malloc(len);
	memcpy_s(c,len,infile,len);
	/* Flip letters */
	c[len-2] = 'm';
	c[len-3] = 'b';
	c[len-4] = 'g';
	/* Return */
	return c;
}

/* Program entry */
int main(int argn,char **argv)
{
	image img;
	char *out;
	int cmode;
	int i;
	/* No ops? */
	if(argn == 1)
	{
		printf("Usage: (flags) [infile]\n");
		return 0;
	}
	/* Process ops */
	for(i = 0;i < argn;i++)
		gbabmp_option(argv[i]);
	/* Read the input file */
	gbaimg_load(&img,argv[argn-1]);
	/* Generate output name */
	out = gbabmp_outfile(argv[argn-1]);
	/* Default Convert Mode */
	if(img.model == IMAGE_INDEX_MODEL)
	{
		cmode = METHOD_8;
		gbabmp_trans_color(&img);
	}
	else
	{
		cmode = METHOD_16;
	}
	/* Convert */
	cmode = gbaconv_run(&img,cmode,option_cmode);
	/* Display */
	printf("%s [%s] (%d,%d) -> [%s] (%dc,%dc)\n",out,gbaimg_name(img.model),img.w,img.h,gbaexp_name(cmode),img.w/8,img.h/8); 
	/* And then convert */
	gbaexp_conv(&img,out,cmode);
	/* End */
	free(out);
	return 0;
}
